/*
 * $Id: sched.c,v 1.8 2006/03/30 17:58:58 lorenzo Exp $
 * 
 * Copyright (C) 2006 RIPE NCC
 * 
 * Original developer: Lorenzo Colitti <lorenzo@ripe.net>
 * Contributor(s):
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include <pthread.h>

#include "pinger.h"

/** Conditional variable for when a thread finishes */
pthread_cond_t thread_done = PTHREAD_COND_INITIALIZER;

/** Lock to protect thread_done */
pthread_mutex_t thread_done_lock = PTHREAD_MUTEX_INITIALIZER;

/**
 * Sleeps for the given number of microseconds using |nanosleep()|
 * @param us number of microseconds to sleep
 */
int sleepinterval(int us) {
	struct timespec ts;
	ts.tv_nsec = (us % 1000000) * 1000;
	ts.tv_sec = us / 1000000;
	return nanosleep(&ts, NULL);
}

/**
 * Prints a message horizontally offset according to thread number
 * @param threadslot number of tabs to offset
 * @param msg message
 */
void threadprint(int threadslot, char *msg) {
	int i;
	printf("\t\t");
	for(i = 0; i < threadslot; i++)
		printf("\t\t");

	printf("%s\n", msg);
}

void *recv_thread_func(void *args) {
	struct recvparm *recvparm = (struct recvparm *) args;
	struct target *target = NULL;
	struct ping_reply reply;
	struct targetlist targetlist = recvparm->targetlist;

	while(1) {
		target = recvpacket(recvparm->sock, &reply, recvparm->pid, targetlist);
		while(target == NULL) {
			target = recvpacket(recvparm->sock, &reply, recvparm->pid, targetlist);
		}

		/* Write result */
		pthread_mutex_lock(&target->lock);
		target->replies[reply.seq] = reply;
		pthread_mutex_unlock(&target->lock);
	}
}

/**
 * Sends a specified number of pings to a given host
 * @param args a |struct job| specifying job parameters
 */
void *send_thread_func(void *args) {
	struct job *job = (struct job *) args;
	int i, ret;

	/* Prepare result structure */
	pthread_mutex_lock(&job->target->lock);
	job->target->nreceived = 0;
	job->target->nsent = 0;
	job->target->replies = calloc(job->numpings, sizeof(struct ping_reply));
	if(! job->target->replies) {
		pthread_mutex_unlock(&job->target->lock);
		die("calloc");
	}

	for(i = 0; i < job->numpings; i++)
		job->target->replies[i].type = ICMP_NONE;

	pthread_mutex_unlock(&job->target->lock);
	 
	for(i = 0; i < job->numpings; i++) {
		pthread_mutex_lock(&job->target->lock);
		job->target->nsent++;
		if(job->udpmode) {
			ret = sendudp(job->sock, job->target, i);
		} else {
			ret = sendping(job->sock, job->target, i, job->id);
		}

		pthread_mutex_unlock(&job->target->lock);

		if(ret == 0) {
			perror("sendping");
		}
		sleep(1);
	}

	/* Signal completion */
	pthread_mutex_lock(&job->lock);
	job->running = 0;
	pthread_mutex_unlock(&job->lock);

	pthread_mutex_lock(&thread_done_lock);
	pthread_cond_broadcast(&thread_done);
	pthread_mutex_unlock(&thread_done_lock);

	return ((void*) 0);
}

int get_next_free_thread(struct job *jobs, int numjobs) {
	int i, nextslot = -1;

	/* Don't want any threads to finish now */
	pthread_mutex_lock(&thread_done_lock);

	while(nextslot == -1) {
		/* Scan array looking for a free thread */
		for(i = 0; (i < numjobs) && (nextslot == -1); i++) {
			pthread_mutex_lock(&jobs[i].lock);

			if(jobs[i].running == 0) {
				nextslot = i;
				pthread_mutex_unlock(&jobs[i].lock);
				break;
			}

			pthread_mutex_unlock(&jobs[i].lock);
		}

		if(nextslot == -1) {
			/* No free threads, wait for one */
			pthread_cond_wait(&thread_done, &thread_done_lock);
		}
	}

	/* Unlock mutex */
	pthread_mutex_unlock(&thread_done_lock);

	return nextslot;
}

