# Copyright (c) 1996, 1997                      The TERENA Association
# Copyright (c) 1998, 1999                      RIPE NCC
#
# All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of the author not be
# used in advertising or publicity pertaining to distribution of the
# software without specific, written prior permission.
#
# THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
# ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
# AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
# AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
# OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#------------------------------------------------------------------------------
# Filename          :   Check.pm
# Purpose           :   Given an array of pointers to hashes each containing an
#                       extracted request form template, goes through each
#                       of the hashes attaching problem codes dependent on the
#                       field values. Problem codes for between-template problems
#                       are then attached. Only one pass-through is required.
# Author            :   Lee Wilmot
# Date              :   971114
# Language Version  :   Perl 5.003_07
# OSs Tested        :   BSD
# Command Line      :   Nothing executable from the command line.
# Input Files       :   NONE
# Output Files      :   NONE
# External Programs :   $REG_COM
# Problems          :
# To Do             :   With regard to the database, it would be nice to insert
#                       a section in which the network and person templates are 
#                       TEST-submitted to the database, and a report of any problems
#                       is returned for inclusion in the final report. This is not
#                       easily possible at the moment.
# Comments          :   The 'current usage' and 'addressing plan' checkers are
#                       very similar at many points. Unfortunately, they we're
#                       just different enough to make it very difficult to
#                       factorize the code.
# Description       :
#
# All of the routines in this package are concerned with analysing
# each of the templates extracted by Extract.pm for problems.
# It is thus reliant on the format used by Extract.pm.
# This package adds error/warning/remark codes, which can then
# be interpreted by following report packages. 
#
# General operation is: pass through each hash, adding problem codes
# as we go. We also make notes ( in %check::need ) of data which
# means that we expect to see something else in another template.
# By also noting things that we've found that other templates might
# require (in %check::found ), we avoid a double pass through the
# data. 
# 
# After the last template is checked, we compare what we need with what
# we found, and attach problem codes to the meta template dependent on
# the results. They go in the meta-template, because they're not specifically
# associated with one particular template.
#
# We also keep track of numbers and types of templates found and can therefore
# report on missing templates, extra (unwanted) templates etc.
#
# ### DATA STRUCTURES ###
#
# Two hashes are involved in tracking certain important data which we have seen
# and which we need to see.
#
# ## %check::found ###

# Possible Keys                 Possible Values

# Any VALUE from the TNAMES hash, e.g.      Numeric count of number of templates
# OVERVIEW OF ORGANIZATION TEMPLATE     of that name found.
#                       undef = no template of that name found.
#
# $F{PIREQ}}                    0 -> pireq=yes found in network template.
#                               1 -> pireq=no found in request overview.
#                               undef -> no entry found for pireq.
#
# $F{OVRIMM}                    Takes the value of $F{OVRIMM} in overview
#                               of request template.
# $F{OVRYR1}                    Year 1.
# $F{OVRYR2}                    Year 2.
#
# $F{OVRSUBIMM}                 Takes the value of no-of-subnets-immediate
#                               in overview of request template.
# $F{OVRSUBYR1}                 Year 1.
# $F{OVRSUBYR2}                 Year 2.
#
# $F{ADRSUBIMM}                 Counts no of subnets used immediately which
#                               also have > 0 addresses used immediately
# $F{ADRSUBYR1}                 Year 1.
# $F{ADRSUBYR2}                 Year 2.
#
# $F{ADRIMM}                    Takes the value of the total immediate usage
#                               from the addressing plan.
# $F{ADRYR1}                    Year 1
# $F{ADRYR2}                    Year 2
#
# $F{STATUS}                    Takes the value of the status field in
#                               the network template.
# $F{NICHDL}                    Any NIC handles found in person templates
#                               on the form separated by $FORM_FS, or
#                               undef -> none found
#
# ## %check::need ##
#
# Possible Keys                 Possible Values
#
# $F{NICHDL}                    $FORM_FS separated nic-hdl's taken from
#                               the admin-c and tech-c contact person
#                               fields in the network template.
#
#
# ### DATA FORMAT EXPECTATIONS ###
#
# 1)    Meta template is first template extracted. It's at $TEMPLATES[0].
# 2)    Multiple lines on the form are translated into $FORM_FS separated lines
#       in the field value of an extracted hash.
# 3)    Whitespace has been removed from the front and end of field names
#       and field values.
# 4)    Subnet lines in a plan template are given a numeric key, corresponding
#       to the line number of the subnet on the plan. Their values are separated
#       by $PLAN_FS.
# 5)    When Extract.pm finds lines in a template which it does not
#       expect, they're added to the REMAINDER field of the template.
# 
# ### FIELD NAMES ADDED TO TEMPLATES BY THIS PACKAGE ###
#
# $F{PROBLEMS}
#
# Added by &prob. Holds an array of problems descriptions.
# Within each problem description is a problem code and a set of arguments,
# all separated by $PROB_FS.

############################ PACKAGE INTERFACE ###############################
#
# Please see the package concerned for descriptions of imported symbols.

package Check;

use strict;

BEGIN {

    use vars qw ( @ISA @EXPORT_OK );

    use Exporter();

    @ISA = qw( Exporter );

    @EXPORT_OK = qw(
        &check &prob
    );
}

# MGTM added, to decide if should show a given problem: REPORT_TYPE_FLAG, DONT_REPORT_THESE_FOR_WEB

use RobotConfig qw(     
    :REGULAR_EXPRESSIONS

    %F $FORM_FS $PROB_FS $PLAN_FS $VISIBLE_FS
    @F_MUST_BE_INTEGER @F_MULTIPLE_ALLOWED %F_MISSPELLING
    @F_MUST_BE_SINGLE_VALUE @F_MAY_USE_SLASH_NOTATION

    $IMMEDIATE_OVERALL_EFFICIENCY_THRESHOLD 
    $YEAR1_OVERALL_EFFICIENCY_THRESHOLD $HOLE_WARN_THRESHOLD
    $OVERALL_EFFICIENCY_TOLERANCE $SUBNET_EFFICIENCY_THRESHOLD
    $PLANLINES_IGNORE_THRESHOLD

    %TEMPLATE_NAMES 

    %DONT_REPORT_THESE_FOR_WEB
    %REPORT_TYPE $REPORT_TYPE_FLAG
);


use Extract qw(
    @TEMPLATES
);

use Misc qw(    
    &fatal &ismask   &quad2int  &int2quad &isnichandle
    &isquad  &mask2int &absent &dprint &subnetsize2mask
    &isemail &iscountry &country_to_code
    &add_to_field &slash2int
);

use TemplateAux qw(
    &template_min_number &template_max_number
    &get_template_key &template_name &is_template_keyfield
);

use Whois qw(
    &indb &handle_in_arin_db
);

########################### PACKAGE BODY ###############################

#   Purpose  :   Main routine for template checking.
#   In       :   VOID
#   Out      :   VOID
#   Comments     :   Analyses @TEMPLATES
#
sub check {

    &dprint("********* CHECK ************");

    # The templates can come in any order and there can be multiple copies of
    # some templates. Therefore, we can't say whether we got what we needed
    # until the end. Therefore, as we go through the templates we keep track of
    # both what we need and what we find (the relevant bits, at least)
    # in these two hashes. This is the case for all restrictions which apply
    # across templates.

    local (%check::found);          
    local (%check::need);

    # Extract pointer to meta template first, because we need to
    # make references to it a lot while checking other templates

    local ($check::meta) = $TEMPLATES[0];

    # If we didn't get a meta template then we didn't get anything at all
    # since this is the first thing extracted.

    &fatal("Meta template not defined, no email info, can't process request. Empty file ?")
        if ( ! defined $check::meta );

    ########### Now process each extracted template in turn  ############

    my $hash_ref;                           # reference to a hash

    my $max_templates_with_this_name;       # each template name has an associated
                                            # maximum number. 

  TEMPLATE:
    foreach $hash_ref (@TEMPLATES) {

        # Get the name and key of the template

        my $template_name = $hash_ref->{$F{TNAME}};

        my $template_key = &get_template_key($template_name);

        # Check we recognise the template name. If we don't
        # signify a problem and don't process it.

        if ( ! $template_key) {
            &prob($hash_ref, "TEMPLATE_IS_UNDEFINED", $template_name);
            next TEMPLATE;
        }
    
        # Track number of templates of each type found.
    
        $check::found{$template_name}++;
    
        # Get maximum number allowed for this template type from
        # the configuration file.
    
        my $max_templates_with_this_name = &template_max_number($template_key);
    
        # Check we're not above this limit already.
    
        if ( $check::found{$template_name} > $max_templates_with_this_name ) {
    
            # If over max, log error, but only the first time
    
            &prob($check::meta, "TOO_MANY_TEMPLATES_OF_TYPE_X", "$template_name", $max_templates_with_this_name )
                if ( $check::found{$template_name} == ($max_templates_with_this_name + 1) );    
            
            # We don't process this template any further.
    
            next TEMPLATE;
        }
    
        # Check contents of each template appropriately for it's type
    
        if    ( $template_key eq 'T_META'       )   { &check_meta($hash_ref);              }
        elsif ( $template_key eq 'T_OVERORG'    )   { &check_org_overview($hash_ref);      }
        elsif ( $template_key eq 'T_REQUESTER'  )   { &check_requester_or_user($hash_ref); }
        elsif ( $template_key eq 'T_USER'       )   { &check_requester_or_user($hash_ref); }
        elsif ( $template_key eq 'T_USAGE'      )   { &check_current_usage($hash_ref);     }
        elsif ( $template_key eq 'T_ADDRESSPLAN')   { &check_address_plan($hash_ref);      }
        elsif ( $template_key eq 'T_PERSON'     )   { &check_person($hash_ref);            }
        elsif ( $template_key eq 'T_OVERREQ'    )   { &check_request_overview($hash_ref);  }
        elsif ( $template_key eq 'T_NETWORK'    )   { &check_network($hash_ref);           }
        else {
            # We should never get here, but you never know.
    
            &prob($hash_ref, "TEMPLATE_IS_UNDEFINED", $template_name);
        }
    }

    # Check for none-template specific problems now that we've been
    # through all the templates

    &check_post_processing_values;
}

########### META TEMPLATE ###########
#
#   Purpose         :   Check meta template for problems
#   Comments        :   
#   In              :   $%: pointer to meta template
#   Out             :   $:  boolean: abandon request ?
#
sub check_meta {

    my $hash_ref = shift @_;

    # Check for disallowed multiple field entries in template.
    
    &multiple_check($hash_ref);     

}

################ OVERVIEW OF ORGANISATION ###############
#
#   Purpose         :   Check the template for problems
#   Side Effects    :   ?
#   Comments        :   
#   In              :   $%: pointer to the hash holding the template
#   Out             :   VOID
#
sub check_org_overview {
    
    my $hash_ref = shift @_;

    # no text -> problem

    &prob($hash_ref, "NO_OVERVIEW_OF_ORGANISATION_IN_REQUEST", &template_name('T_OVERORG') )
        if ( absent($hash_ref->{$F{TEXT}}) );       

}

################ REQUESTER / USER TEMPLATES ###############
#
#   Purpose         :   Check the template for problems
#   Side Effects    :   ?
#   Comments        :   
#   In              :   $%: pointer to the hash holding the template
#   Out             :   VOID
#
sub check_requester_or_user {

    my $hash_ref = shift @_;

    # Check for templates running in to each other

    &multiple_keyfield_check($hash_ref);

    # Check for disallowed multiple entries in templates.

    &multiple_check($hash_ref);     
    
    # Check for common field-name misspellings.

    &misspelling_check($hash_ref);

    # Sometimes they try to use a person template in place
    # of the user template.

    &prob($hash_ref, 'CANT_USE_PERSON_TEMPLATE_HERE', &template_name('T_REQUESTER'), &template_name('T_USER') )
        if ( defined $hash_ref->{$F{PERSON}} );

    ### CHECK SPECIFIC FIELDS ###

    ### NAME ###

    &prob($hash_ref, "FIELD_IS_BLANK", $F{NAME})    
        if ( &absent($hash_ref->{$F{NAME}}) );

    ### ORGANISATION ###

    &prob($hash_ref, "FIELD_IS_BLANK", $F{ORG})
        if ( &absent($hash_ref->{$F{ORG}}) );

    ### COUNTRY ###

    # field absent -> problem

    if ( &absent($hash_ref->{$F{COUNTRY}}) ) {
        &prob($hash_ref, "FIELD_IS_BLANK", $F{COUNTRY});
    }

    # not an acceptable country code -> problem

    elsif ( ! &iscountry($hash_ref->{$F{COUNTRY}}) ) {

        # If it doesn't exist, try and find what they meant instead
        # of just saying 'sorry don't know it'.

        my $possible_codes = &country_to_code($hash_ref->{$F{COUNTRY}});

        # The problem text is passed some possibilities, if there ARE any.

        if ( $possible_codes =~ /^\s*$/ ) {
            &prob($hash_ref, "INVALID_COUNTRY_CODE", $hash_ref->{$F{COUNTRY}}, " " );
        }

        else {
            &prob($hash_ref, "INVALID_COUNTRY_CODE", $hash_ref->{$F{COUNTRY}}, "Suggested possibilities: $possible_codes." );
        }
    }

    ### EMAIL ###

    # field absent in requester template -> problem
    # the user might need the addresses before he has an email address :)

    &prob($hash_ref, "FIELD_IS_BLANK", $F{EMAIL})
        if ( $hash_ref->{$F{TNAME}} eq &template_name('T_REQUESTER') && &absent($hash_ref->{$F{EMAIL}}) );

    # Any of the addresses non-RFC822 value -> problem

    if (! &absent($hash_ref->{$F{EMAIL}}) ) {

        my $address_to_check;

        foreach $address_to_check (split $FORM_FS, $hash_ref->{$F{EMAIL}}) {

            &prob($hash_ref, "EMAIL_ADDRESS_FORMAT_PROBLEM", $address_to_check, $F{EMAIL}) 
                if ( ! &isemail($address_to_check) );
        }

    }
}

################ CURRENT USAGE PLAN ###############
#
#   Purpose         :   Check the template for problems
#   Side Effects    :   ?
#   Comments        :   
#   In              :   $%: pointer to the hash holding the template
#   Out             :   VOID
#
sub check_current_usage {

    my $hash_ref = shift @_;    

    my $expected_address;       # Which address we expect at the current line
                                # based on last prefix and last subnet size

    my $line;                   # Which line of the usage plan we're on

    my $sizetot = 0;            # Hold cumulative totals from plan
    my $immtot = 0;
    my $yr1tot = 0;
    my $yr2tot = 0; 

    my %prefix_count;		# Hold all prefixes seen so far.
                                # Used to check for duplicates.
    my %prefix_lines;		# Hold line numbers of prefixes found
    my %prefix_sizes;		# Hold sizes of prefixes found

    my $lines_found = 0;        # Used to count no of lines seen

    my $mask_int;

    my $relative_prefix_flag = 0;       # Flags if relative prefixes being used or not

    my ($prefix, $prefix_start_int, $prefix_end_int);          # Temp var for loops

				# Used in comparing current prefix to previous prefixes
    my ($other_prefix, $other_prefix_start, $other_prefix_end);

    # Check the plan isn't too big for analysis

    return 
        if ( &plan_too_big($hash_ref) );

    # Check for possible missing lines in the remainder field before
    # going through each subnet. Any lines which Extract.pm didn't 
    # recognise as a plan line are added to the REMAINDER field, so
    # at this point we can check for near-misses.
    # If we had one or more near-misses, we put in a problem and stop
    # processing the plan at this point, because missing subnets just
    # result in a waterfall of errors.

    if ( &subnets_in_remainder($hash_ref) ) {
        &prob($hash_ref, "ABANDONED_PLAN"); 
        return;
    }

    # Check for disallowed multiple field entries in template.

    &multiple_check($hash_ref);

    # The main checks...

    foreach $line (sort mixsort keys %$hash_ref) {

        # Ignore none-numberic keys. All subnets in the we're given a
        # number by Extract.pm, in the order that they were extracted.

        next if ( $line !~ /^\d+$/ );

        # Get next line

        my ($prefix, $mask, $size, $imm, $yr1, $yr2, $desc)
            = split /$PLAN_FS/, $hash_ref->{$line};


        $lines_found++;                     # Record having seen a line
    
        # Check if prefix looks relative. It should be absolute, since this
        # is a current usage plan.
    
        $relative_prefix_flag = 1   
            if ( $prefix =~ /^\s*0./ );     

        # Check description field for the subnet not blank.
    
        &prob($hash_ref, "NO_DESCRIPTION_FIELD_IN_PLAN", $prefix, $line)
            if ( $desc =~ /^\s*$/ );
    
        ############### Do MASK/SIZE related checks #################

        # If subnet mask has got missing digits, add ".0"'s

        if ( &isquad($mask) > 1 ) {
            my $newmask = $mask.(".0" x (4 -&isquad($mask)));

            if ( $newmask ne $mask ) {
                &prob($hash_ref, "EXTENDED_DOTTED_QUAD", $line, $mask, $newmask);
                $mask = $newmask;
            }
        }
        # If subnet mask has a valid value, check it for a few problems
    
        if ( ! &ismask($mask) ) {

            # Otherwise, signify a problem and try and suggest a good value
            # using the subnet size. 
    
            &bad_subnet_mask_problem($hash_ref, $size, $mask, $prefix, $line);
    
        }
        else {

            $mask_int = &mask2int($mask);

            # We add 2 to the size of the subnet if the mask is valid and 
            # it looks like they've just taken off the 2 unusable addresses.
    
            $size += 2   
                if ( $mask_int == ($size + 2) );
    
            # Check subnet mask = (size specified OR size specified + 2). People 
            # often remove the 2 unusable addresses for the subnet.
    
            &prob($hash_ref, "SUBNET_SIZE_DIFFERENT_FROM_SUBNET_MASK", $size, $mask, $prefix, $line)    
                if ( $size != $mask_int );
        }
    
        ################# PREFIX DEPENDENT CHECKS ##################

        # Add '.0's to subnet prefix if necessary

        if ( &isquad($prefix) > 1 ) {       # if they've given at least 2 already

            my $newprefix = $prefix.(".0" x (4 -&isquad($prefix)));

            # If we do this, warn user
            if ( $newprefix ne $prefix ) {
                &prob($hash_ref, "EXTENDED_DOTTED_QUAD", $line, $prefix, $
newprefix);
                $prefix = $newprefix;
            }
        }

        $prefix_count{$prefix} ++;         # Record seeing this prefix

        if ( &isquad($prefix) == 4 ) {

            $prefix_start_int = &quad2int($prefix);     

            # Check for this subnet overlapping with previously seen ones

            $prefix_end_int = $prefix_start_int + $size;

            foreach $other_prefix ( sort keys %prefix_sizes ) {

                $other_prefix_start = &quad2int($other_prefix);
                $other_prefix_end = $other_prefix_start + $prefix_sizes{$other_prefix};

                if ( ( $prefix_start_int > $other_prefix_start && $prefix_start_int < $other_prefix_end ) 
        	   || ( $prefix_end_int > $other_prefix_start && $prefix_end_int < $other_prefix_end ) ) {
                    &prob($hash_ref, "TWO_SUBNETS_IN_PLAN_OVERLAP", $other_prefix, $prefix_lines{$other_prefix}, $prefix, $line);
			    
                }
            }
        }
        else {
            &prob($hash_ref, "INVALID_DOTTED_QUAD", $prefix, $line);
        }

	# Add current one it so it's compared with future subnets

	$prefix_sizes{$prefix} = $size;
	$prefix_lines{$prefix} = $line;
    
        # Update running totals of addresses.
        
        $sizetot += $size; $immtot += $imm; $yr1tot += $yr1; $yr2tot += $yr2;

        # Check for invalid bit-boundary
        # Only if both the bit-boundary and the mask were OK

        if ( &isquad($prefix) == 4 && &ismask($mask)  ) {

            my $boundary_value = $prefix_start_int / $mask_int;
    
            # Give a problem if it's NOT an integer.
    
             &prob($hash_ref, "NOT_A_VALID_BIT_BOUNDARY", $prefix, $mask, $mask_int)   
                    if ( ($boundary_value - int ($boundary_value) ) != 0 );

        }
   
    }
    
    ########### WE'VE SEEN ALL OF THE SUBNETS NOW #############

    # If we actually found any subnets...

    my $this_prefix;

    if ( $lines_found > 0 ) {

        # Check each prefix only seen once

        foreach $this_prefix (sort keys %prefix_count) {
	    if ( $prefix_count{$this_prefix} > 1 ) {
		&prob($hash_ref, "SUBNET_PREFIX_IS_DUPLICATED_IN_PLAN", $this_prefix, $prefix_count{$this_prefix});
	    }
        }
    
        # Warn if it looks like they're using relative prefixes
    
        &prob($hash_ref, "LOOKS_LIKE_RELATIVE_PREFIXES")
            if ( $relative_prefix_flag );

        # Analyse the totals line at the end of the plan 
        # if there is one.

        if ( &absent($hash_ref->{$F{APTOTALS}}) ) {

            # If there was only ONE subnet, we
            # don't mind if they skip the totals line.

            &prob($hash_ref, "NO_TOTALS_LINE_AT_BOTTOM_OF_PLAN")
                if ($lines_found > 1);

        }
        else {
    
            # Grab the totals line
    
            my ($sizetot_on_form, $immtot_on_form, $yr1tot_on_form, $yr2tot_on_form) =
                split /$PLAN_FS/, $hash_ref->{$F{APTOTALS}};

            # Check totals in plan = totals we calculated

            &prob($hash_ref, "TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", $sizetot_on_form, $sizetot)   
                if ( $sizetot_on_form != $sizetot );
        
            &prob($hash_ref, "PERIOD_TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", "immediate", $immtot_on_form, $immtot) 
                if ( $immtot_on_form != $immtot );
        
            &prob($hash_ref, "PERIOD_TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", "year 1", $yr1tot_on_form, $yr1tot) 
                if ( $yr1tot_on_form != $yr1tot );
        
            &prob($hash_ref, "PERIOD_TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", "year 2", $yr2tot_on_form, $yr2tot) 
                if ( $yr2tot_on_form != $yr2tot );
        }
    }
}

################ REQUEST OVERVIEW TEMPLATE ###############
#
#   Purpose         :   Check the template for problems
#   Side Effects    :   ?
#   Comments        :   
#   In          :   $%: pointer to the hash holding the template
#   Out         :   VOID
#
sub check_request_overview {

    my $hash_ref = shift @_;

    # Check for templates running in to each other

    &multiple_keyfield_check($hash_ref);

    # Check for common misspellings in field names

    &misspelling_check($hash_ref);  

    # Check for disallowed multiple entries in templates

    &multiple_check($hash_ref);

    # Check for multiple values where it's not allowed

    &single_value_check($hash_ref);

    # Check fields which should be integers

    &integer_check($hash_ref);

    ### CHECK SPECIFIC FIELDS ###

    ### INET CONNECT ###

    &prob($hash_ref, "FIELD_IS_BLANK", $F{INETCONNECT})
        if ( &absent($hash_ref->{$F{INETCONNECT}}) );

    ### PRIVATE CONSIDERED ###

    # absent -> problem

    if ( &absent($hash_ref->{$F{PRVCONSID}}) ) {
        &prob($hash_ref, "FIELD_IS_BLANK", $F{PRVCONSID});
    }

    # a negative response (i.e. private NOT considered) -> problem

    elsif ( $hash_ref->{$F{PRVCONSID}} !~ /$PRVCONSID_REG/i ) {
        &prob( $hash_ref, "PRIVATE_ADDRESSES_NOT_CONSIDERED", $F{PRVCONSID} );
    }

    ### COUNTRY-NET ###

    # absent -> problem

    if ( &absent($hash_ref->{$F{CNTRYNET}}) ) {
        &prob($hash_ref, "FIELD_IS_BLANK", $F{CNTRYNET});
    }

    # check all are valid iso3166

    else {

        # Lee 19980911 The CNTRYNET field can have multiple lines.
        # Add an outer loop to split up the lines and check each
        # separately.

        my ( $line, $code );
        foreach $line (split /$FORM_FS/, $hash_ref->{$F{CNTRYNET}} ) {

            # Lee 19980917
            # Changed splitting pattern to cope with comma's,
            # multiple space etc

            foreach $code (split /[^A-Z]+/i, $line ) {

                # Necessary because the split can produce a
                # null string. Probably a better match exists.

                next if ( ! $code );

                if ( ! &iscountry($code) ) {
                    
                    # If it doesn't exist, try and find what they meant instead
                    # of just saying 'sorry don't know it'.
                    
                    my $possible_codes = &country_to_code($code);
                    
                    # The problem text is passed some possibilities, if there ARE any.
                    
                    if ( $possible_codes =~ /^\s*$/ ) {
                        &prob($hash_ref, "INVALID_COUNTRY_CODE", $code, " " );
                    }
                    else {
                        &prob($hash_ref, "INVALID_COUNTRY_CODE", $code, "Suggested possibilities: $possible_codes." );
                    }
                }
            }
        }
    }

    ### PI-REQUESTED ###

    # absent -> problem

    if ( &absent($hash_ref->{$F{PIREQ}}) ) {
        &prob($hash_ref, "FIELD_IS_BLANK", $F{PIREQ});
    }

    # If positive response found, mark down that we've found it
    # This is checked later against what they said in the
    # network template.

    elsif ( $hash_ref->{$F{PIREQ}} =~ /$PIREQPOS_REG/i ) {   
        $check::found{$F{PIREQ}} = 1;
    }

    # If negative response found, mark down that we've found it
    # This is checked later against what they said in the
    # network template.

    elsif ( $hash_ref->{$F{PIREQ}} =~ /$PIREQNEG_REG/i ) {
        $check::found{$F{PIREQ}} = 0;
    }

    # Otherwise it's an unclear response -> problem

    else {
        &prob($hash_ref, "PI_REQUESTED_FIELD_MUST_BE_YES_OR_NO", $F{PIREQ});
    }

    # Record some values to compare with addressing plan totals later

    $check::found{$F{RSIZE}} = $hash_ref->{$F{RSIZE}};

    $check::found{$F{OVRIMM}} = $hash_ref->{$F{OVRIMM}};
    $check::found{$F{OVRYR1}} = $hash_ref->{$F{OVRYR1}};
    $check::found{$F{OVRYR2}} = $hash_ref->{$F{OVRYR2}};

    $check::found{$F{OVRSUBIMM}} = $hash_ref->{$F{OVRSUBIMM}};
    $check::found{$F{OVRSUBYR1}} = $hash_ref->{$F{OVRSUBYR1}};
    $check::found{$F{OVRSUBYR2}} = $hash_ref->{$F{OVRSUBYR2}};
}


################ ADDRESSING PLAN ###############
#
#   Purpose         :   Check the template for problems.
#   Side Effects    :   ?
#   Comments        :   
#   In          :   $%: pointer to the hash holding the template
#   Out         :   VOID
#

sub check_address_plan {

    my $hash_ref = shift @_;            

    my $line = 0;               # Which line of the addressing plan we're on.

    my $sizetot = 0;            # Hold cumulative totals from plan
    my $immtot = 0;
    my $yr1tot = 0;
    my $yr2tot = 0; 
    my $holetot = 0;            # (this to add up addresses wasted in holes)

    my %prefix_count;		# Hold all prefixes seen so far.
                                # used to check for duplicates
    my %prefix_lines;		# Hold line numbers of prefixes found
    my %prefix_sizes;		# Hold sizes of prefixes found
    my %prefix_ok;              # Hold whether each prefix was OK.

    my $first_prefix = "255.255.255.255";  # For recording the value of the lowest prefix
                                           # found on the plan

    my $last_address = 0;       # For recording the value of the last address used on the
                                # plan.

    my $lines_found = 0;        # Used to count no of lines seen

    my $not_relative_flag = 0;  # Flags if relative prefixes being used or not

    my ($prefix);		# Vars for storing the current subnet's prefix
                                # and its integer equivalents
    my ($prefix_start_int, $prefix_end_int);    
            
                                # Record the previous prefix and it's last address
    my ($previous_prefix, $previous_prefix_end_int);

    my $mask_int;               # Hold the integer value of the current mask.

				# Used in comparing current prefix to previous prefixes
    my ($other_prefix, $other_prefix_start, $other_prefix_end);

    # Check first the plan isn't too big for analysis.

    return
        if ( &plan_too_big($hash_ref) );

    $check::found{$F{ADRSUBIMM}} = 0;
    $check::found{$F{ADRSUBYR1}} = 0;
    $check::found{$F{ADRSUBYR2}} = 0;

    # Check for disallowed multiple field entries in template.

    &multiple_check($hash_ref);
    
    # Check for possible missing lines in the remainder field before
    # going through each subnet. Any lines which Extract.pm didn't 
    # recognise as a plan line are added to the REMAINDER field, so
    # at this point we can check for near-misses.
    # If we had one or more near-misses, we put in a problem and stop
    # processing the plan at this point, because missing subnets just
    # result in a waterfall of errors.

    if ( &subnets_in_remainder($hash_ref) ) {
        &prob($hash_ref, "ABANDONED_PLAN"); 
        return;
    }

    # Main loop: go through each line of plan looking for problems
    # and making totals

    foreach $line (sort mixsort keys %$hash_ref) {

        # Ignore none-numberic keys. All subnets in the we're given a
        # number by Extract.pm, in the order that they were extracted.
    
        next if ( $line !~ /^\d+$/ );
    
        # Get next line
    
        my ($prefix, $mask, $size, $imm, $yr1, $yr2, $desc)
            = split /$PLAN_FS/, $hash_ref->{$line};

        $lines_found ++;                    # Record having seen a line

	################# MISCELLANEOUS CHECKS ##############

        # Check if prefix looks relative. It should be NOT BE ABSOLUTE,
        # since this is a current usage plan.
    
        $not_relative_flag = 1
            if ( $prefix !~ /^\s*0./ );         

        # Description blank -> problem
    
        &prob($hash_ref, "NO_DESCRIPTION_FIELD_IN_PLAN", $prefix, $line)
            if ( $desc =~ /^\s*$/ );


	############### Do MASK/SIZE related checks #################

        # If subnet mask has got missing digits, add ".0"'s

        if ( &isquad($mask) > 1 ) {
            my $newmask = $mask.(".0" x (4 -&isquad($mask)));

            if ( $newmask ne $mask ) {
                &prob($hash_ref, "EXTENDED_DOTTED_QUAD", $line, $mask, $newmask);
                $mask = $newmask;
            }
        }

        # If subnet mask has a valid value, check it for a few problems
    
        if ( ! &ismask($mask) ) {

            # Otherwise, signify a problem and try and suggest a good value
            # using the subnet size. 
    
            &bad_subnet_mask_problem($hash_ref, $size, $mask, $prefix, $line);

    
        }
        else {

            $mask_int = &mask2int($mask);
    
            # We add 2 to the size of the subnet if the mask is valid and 
            # it looks like they've just taken off the 2 unusable addresses.
    
            $size += 2   
                if ( $mask_int == ($size + 2) );
    
            # Check subnet mask = (size specified OR size specified + 2). People 
            # often remove the 2 unusable addresses for the subnet.
    
            &prob($hash_ref, "SUBNET_SIZE_DIFFERENT_FROM_SUBNET_MASK", $size, $mask, $prefix, $line)    
                if ( $size != $mask_int );

        }
 
	################# PREFIX DEPENDENT CHECKS ##################

	# Add '.0's to subnet prefix if necessary
        if ( &isquad($prefix) > 1 ) {

            my $newprefix = $prefix.(".0" x (4 -&isquad($prefix)));

            # If we do this, warn user
            if ( $newprefix ne $prefix ) {
                &prob($hash_ref, "EXTENDED_DOTTED_QUAD", $line, $prefix, $newprefix);
                $prefix = $newprefix;
            }
        }

        $prefix_count{$prefix} ++;         # Record seeing this prefix

	if ( &isquad($prefix) == 4 ) {

            # Convert it for use later don't set $prefix_end_int yet
            # because we're not sure of size value

	    $prefix_start_int = &quad2int($prefix);	

            # Do some checks against the previous prefix, if it was OK.

            if ( $prefix_ok{ $previous_prefix } ) {

                # Add to holes total if there's one here. A hole is where there's a gap
                # between where the last subnet ends and this one starts.
                # ($prefix_end_int is still set at this point to the value from the
                # last subnet seen)
    
                $holetot += (  $prefix_start_int - $previous_prefix_end_int )
                    if ( $prefix_start_int - $previous_prefix_end_int > 0 );

		# Check for decreasing prefixes

		&prob($hash_ref, "SUBNETS_IN_WRONG_ORDER", $prefix, $line, $previous_prefix)
                    if ( $prefix_start_int < &quad2int($previous_prefix) );
            }

            # If this prefix is less than the currently set first prefix (which will
            # only happen if the prefixes are in the wrong order), make it the first

            $first_prefix = $prefix
		if ( $prefix_start_int < &quad2int($first_prefix));

            # Calc the last address in this subnet

	    $prefix_end_int = $prefix_start_int + $size;

            # Update the plan-scope last address if this is the biggest seen so far

            $last_address = $prefix_end_int
		if ( $prefix_end_int > $last_address );

            # Check for this subnet overlapping with previously seen ones

            foreach $other_prefix ( sort keys %prefix_sizes ) {

		if ( $prefix_ok {$other_prefix} ) {
		    $other_prefix_start = &quad2int($other_prefix);
		    $other_prefix_end = $other_prefix_start + $prefix_sizes{$other_prefix};

		    if ( ( $prefix_start_int > $other_prefix_start && $prefix_start_int < $other_prefix_end ) 
		   || ( $prefix_end_int > $other_prefix_start && $prefix_end_int < $other_prefix_end ) ) {
			&prob($hash_ref, "TWO_SUBNETS_IN_PLAN_OVERLAP", $other_prefix, $prefix_lines{$other_prefix}, $prefix, $line);
			    
		    }
		}
            }

	    # Record prefix for use next time
    
	    $previous_prefix_end_int = $prefix_end_int;
	
            # Signify to next round that it can use expected values

            $prefix_ok{$prefix} = 1;

        }

	# Prefix NOT OK. Signify the fact.

	else {

            &prob($hash_ref, "INVALID_DOTTED_QUAD", $prefix, $line);

            # Signify to next round not to try and check for expected values
            $prefix_ok{$prefix} = 0;
	}

	# Add current one it so it's compared with future subnets.
	# We wait till here so we're sure size is set OK.

        $previous_prefix = $prefix;
	$prefix_sizes{$prefix} = $size;
	$prefix_lines{$prefix} = $line;

	################### USAGE / EFFICIENCY ###################
    
        # Usage values increase -> problem
    
        &prob($hash_ref, "SUBNET_USAGE_DECREASES", $prefix, $line) 
            if ( $imm > $yr1 || $yr1 > $yr2 );
    
        # Check efficiency of subnet usage. The largest usage (immediate, 
        # yr1 or yr2) must be at least $SUBNET_EFFICIENCY_THRESHOLD% of 
        # the subnet size. Need explicit check on size to avoid /0 error.
    
        if ( $size > 0 ) {


        }
        else { 
           
            &prob($hash_ref, "FIELD_MUST_BE_GREATER_THAN_ZERO", $F{SUBNETSIZE}, $prefix, $line);
        }
    
        # Update running totals of addresses
    
        $sizetot += $size; $immtot += $imm; $yr1tot += $yr1; $yr2tot += $yr2;
    
        # Update number of subnets found. These are compared later to the
        # request overview template
    
        $check::found{$F{ADRSUBIMM}} ++
            if ( defined $imm && $imm > 0 );                
    
        $check::found{$F{ADRSUBYR1}}++      
            if ( defined $yr1 && $yr1 > 0 );
    
        $check::found{$F{ADRSUBYR2}}++      
            if ( defined $yr2 && $yr2 > 0 );


        # Bit boundary

        if ( $prefix_ok{$prefix} && &ismask($mask) ) {

            # Needs to be last cos depends on valid prefix AND mask

            # Check that a valid bit boundary is being used. If so, the prefix
            # divided by the mask size should yield an integer.
   
            my $boundary_value = $prefix_start_int / $mask_int;
    
            # Give a problem if it's NOT an integer.
    
             &prob($hash_ref, "NOT_A_VALID_BIT_BOUNDARY", $prefix, $mask, $mask_int)   
                    if ( ($boundary_value - int ($boundary_value) ) != 0 );

        }
    }

    ########### WE'VE SEEN ALL OF THE SUBNETS NOW #############

    # If we actually found any subnets, check other stuff out

    if ( $lines_found == 0 ) {
        &prob($hash_ref, "NO_SUBNETS_FOUND_IN_PLAN");
    }

    else {
    
        # Check each prefix was only seen once
    
        foreach $prefix (sort keys %prefix_count) {
        
            &prob($hash_ref, "SUBNET_PREFIX_IS_DUPLICATED_IN_PLAN", $prefix, $prefix_count{$prefix})
                if ( $prefix_count{$prefix} > 1 );
        }
        
        #Warn if it looks like they're using absolute prefixes
        
        &prob($hash_ref, "LOOKS_LIKE_ABSOLUTE_PREFIXES")
            if ( $not_relative_flag );                          
    
        # Check overall efficiencies OK, and amount of wasted hole space
    
        if ( $sizetot > 0 ) {           #avoid divide-by-zero error


            # Too much wasted hole space -> problem        
            # Only if no prefixes occured twice & all prefixes we're OK.
            if ( scalar ( grep { $prefix_count{$_} > 1 } keys %prefix_count) == 0 &&
                scalar (grep { $prefix_ok{$_} == 0 } keys %prefix_ok ) == 0) {


                &prob($hash_ref, "ADDRESSES_WASTED_BY_HOLES_IN_PLAN", $holetot, int (($holetot/($holetot+$sizetot))*100))
                    if ( (($holetot / ($holetot + $sizetot)) *100) > $HOLE_WARN_THRESHOLD );
            }
        }

        # Analyse the totals line at the end of the plan. If there's only
        # a single line, use that as the totals
        
        my ($sizetot_on_form, $immtot_on_form, $yr1tot_on_form, $yr2tot_on_form );
        
        if ( &absent($hash_ref->{$F{APTOTALS}}) ) {
            
            # If there was only ONE subnet, we
            # don't mind if they skip the totals line.
            
            if ( $lines_found > 1 ) {
            
                &prob($hash_ref, "NO_TOTALS_LINE_AT_BOTTOM_OF_PLAN");
            }
            else {
            
                # If we only found one line, then we need to set the total to
                # the values found on that line
            
                ($sizetot_on_form, $immtot_on_form, $yr1tot_on_form, $yr2tot_on_form ) =
                    ( $sizetot, $immtot, $yr1tot, $yr2tot )
            }
        }
        
        else {
            # Grab the totals line
            
            ($sizetot_on_form, $immtot_on_form, $yr1tot_on_form, $yr2tot_on_form) =
                split /$PLAN_FS/, $hash_ref->{$F{APTOTALS}};
            
            # Check totals in plan = totals we calculated...
            
            &prob($hash_ref, "TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", $sizetot_on_form, $sizetot)   
                if ( $sizetot_on_form != $sizetot );
        
            &prob($hash_ref, "PERIOD_TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", "immediate", $immtot_on_form, $immtot) 
                if ( $immtot_on_form != $immtot );
        
            &prob($hash_ref, "PERIOD_TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", "year 1", $yr1tot_on_form, $yr1tot) 
                if ( $yr1tot_on_form != $yr1tot );
        
            &prob($hash_ref, "PERIOD_TOTAL_SIZE_DIFFERENT_FROM_ACTUAL_SIZE", "year 2", $yr2tot_on_form, $yr2tot) 
                if ( $yr2tot_on_form != $yr2tot );
        
        }

        # Check that the addresses covered by the relative prefixes 
        # (last expected address - 1st prefix) = size total

        # Only if there were no invalid prefixes, and no prefixes occured
        # twice.
        if ( scalar ( grep { $prefix_ok{$_} == 0 } keys %prefix_ok ) == 0 &&
            scalar ( grep { $prefix_count{$_} > 1 } keys %prefix_count) == 0) {

            my ($addresses_covered) = $last_address - &quad2int($first_prefix);
        
            &prob($hash_ref, "ADDRESSES_COVERED_IN_PLAN_DIFFERENT_FROM_ADDRESSES_REQUESTED", "$first_prefix", &int2quad($last_address), $addresses_covered, $sizetot_on_form)
                if ( defined $sizetot_on_form && ( $addresses_covered != $sizetot_on_form ) );
        }
    
        # Record some values for later comparison with the request overview
        
        $check::found{$F{ADRRSIZE}} = $sizetot_on_form;
        $check::found{$F{ADRIMM}} = $immtot_on_form;
        $check::found{$F{ADRYR1}} = $yr1tot_on_form; 
        $check::found{$F{ADRYR2}} = $yr2tot_on_form;
    }
}

################ NETWORK TEMPLATE ###############
#
#   Purpose         :   Check the template for problems
#   Side Effects    :   ?
#   Comments        :   
#   In              :   $%: pointer to the hash holding the template
#   Out             :   VOID
#
sub check_network {

    my $hash_ref = shift @_;

    my $handle;

    # Check for templates running in to each other

    &multiple_keyfield_check($hash_ref);

    # Check for common field-name mis-spellings
    
    &misspelling_check($hash_ref);

    #Check for disallowed multiple entries in templates
    
    &multiple_check($hash_ref);

    ### CHECK SPECIFIC FIELDS ###

    ### NETNAME ###

    # absent -> ERROR.
    # This is ESSENTIAL, so that the request is rejected if there is no
    # netname. It's essential because we use the netname as the basis
    # for checking for duplicate requests later.

    if ( &absent($hash_ref->{$F{NETNAME}}) ) {

        &prob($hash_ref, "FIELD_IS_BLANK", $F{NETNAME});
    }

    # not match required format -> ERROR
    # Again, it's essential that this be an error, for reason
    # described above.

    elsif  ( $hash_ref->{$F{NETNAME}} !~ /$NETNAME_REG/i ) {
        &prob($hash_ref, 'BAD_NETNAME_FORMAT', $F{NETNAME}, $hash_ref->{$F{NETNAME}});
    }

    # has some lowercase letters -> WARNING
    # this is none-essential for the check of duplicate tickets, because
    # we can do 'uc' on it before doing the check

    elsif ( $hash_ref->{$F{NETNAME}} !~ /[A-Za-z0-9-]/ ) {
        &prob($hash_ref, "ILLEGAL_CHARACTERS_IN_NETNAME", $F{NETNAME}, $hash_ref->{$F{NETNAME}});
    }

    ### ADMIN-C ###

    &check_network_contact( $hash_ref, $F{ADMINC} );

    ### TECH-C ###

    &check_network_contact( $hash_ref, $F{TECHC} );
 
    ### STATUS ###

    # Absent -> problem

    if ( &absent($hash_ref->{$F{STATUS}}) ) {

        &prob($hash_ref, "FIELD_IS_BLANK", $F{STATUS});
    }

    # Invalid value -> problem

    elsif ( $hash_ref->{$F{STATUS}} !~ /$STATUS_REG/i ) {

        &prob($hash_ref, "INVALID_VALUE_FOR_STATUS_FIELD", $F{STATUS});
    }

    # Since we got to here, the value is OK so we record it for
    # later comparison with the request overview template.

    else {
        $check::found{$F{STATUS}} = $hash_ref->{$F{STATUS}};

    }
}

################ NETWORK TEMPLATE ###############
#
#   Purpose         :   Check the contact person fields in the network template
#   Comments        :   
#   In              :   $%: pointer to the hash holding the template
#                       $:  the name of the field to check (admin-c, tech-c)
#   Out             :   VOID
#
sub check_network_contact {

    my $hash_ref = shift @_;
    my $field_name = shift @_;

    my $field_value = $hash_ref->{$field_name};

    # Absent -> problem

    if ( &absent( $field_value ) ) {
        &prob( $hash_ref, "FIELD_IS_BLANK", $field_name );
    }

    # Otherwise, check each NIC handle is well-formed, and if it
    # is, record it for later comparison with person objects

    else {

        my $handle;

        # The field could be a list of handles if there is more than
        # one line on the form: do them all.
    
        foreach $handle ( split /$FORM_FS/, $field_value ) {
    
# OLD (19980706)    if ( $handle =~ /$AUTONIC_REG/i || $handle =~ /$NIC_REG/i ) {
    
# NEW
                    if ( $handle =~ /$AUTONIC_REG/i || &isnichandle($handle) ) {

                # Record it
        
                &add_to_field( \$check::need{$F{NICHDL}}, $handle );
            }

            else {
                # Problem !

                &prob($hash_ref, 'INVALID_NIC_HANDLE_IN_INETNUM_OBJECT', $field_name, $handle);
            }
        }
    }
}

################ PEROSN TEMPLATE ###############
#
#   Purpose         :   Check the template for problems
#   Side Effects    :   ?
#   Comments        :   
#   In              :   $%: pointer to the hash holding the template
#   Out             :   VOID
#
sub check_person {

    my $hash_ref = shift @_;

    # Check for templates running in to each other

    &multiple_keyfield_check($hash_ref);

    # Check for common field-name misspellings

    &misspelling_check($hash_ref);
    
    #Check for disallowed multiple entries in templates
    
    &multiple_check($hash_ref);

    ### CHECK SPECIFIC FIELDS ###

    ### KEY FIELD ###    

    my $on_form_key;        # the key of the person/role object found on the form
    my $on_form_object_type;    # the type (person/role) found on the form

    # Check there's either a 'role:' or a 'person:' line. Set the object
    # type and key value appropriately if there is
    
    if ( !&absent($hash_ref->{$F{PERSON}}) ) {
        $on_form_object_type = $F{PERSON};
        $on_form_key = $hash_ref->{$F{PERSON}};
    }

    elsif ( !&absent($hash_ref->{$F{ROLE}}) ) {
        $on_form_object_type = $F{ROLE};
        $on_form_key = $hash_ref->{$F{ROLE}};
    }

    # No 'person:' or 'role:' line -> problem
    # and abandon checking.

    else {

        &prob($hash_ref, "NO_PERSON_OR_ROLE_FIELD_IN_PERSON_TEMPLATE", $F{PERSON}, $F{ROLE}, &template_name('T_PERSON') );

        # Don't check any further: we can't compare the
        # object in the database with the one in the form.

        return; 
    }

    ### NIC-HDL ###

    my $handle = $hash_ref->{$F{NICHDL}};

    # Absent -> problem

    if ( &absent($handle) ) {
        &prob($hash_ref, "FIELD_IS_BLANK", $F{NICHDL});
    }

    # AUTO handle -> record it for later checking

    elsif ( $handle =~ /$AUTONIC_REG/i ) {
        &add_to_field(\$check::found{$F{NICHDL}}, $handle);
    }
    
    # Normal looking NIC handle -> check it out

    # OLD (19980706):    elsif ( $handle =~ /$NIC_REG/i ) {
    # NEW:
    elsif ( &isnichandle($handle) ) {

        # Look for it in the RIPE DB
        # We pass '$on_form_key' as the failure value, so that if whois has a problem
        # we don't later generate a load of errors about missing persons etc.
        
        my $object_found_key = &indb( $on_form_object_type, $on_form_key, $handle, $check::meta );
    
        # If we found an appropriate object, check it's the same one as in the database
        # (Or in any case, that it's key field value matches)
    
        if ( $object_found_key ) {
    
            &prob( $hash_ref, "OBJECT_IN_DB_KEY_MISMATCH", $handle, $object_found_key, $on_form_key, $on_form_object_type )
                if ( $object_found_key ne $on_form_key );
        }
        
        # otherwise, 
        # role object -> failure
        # person object & not in ARIN DB -> failure
    
        elsif ( $on_form_object_type eq $F{ROLE} || ( $on_form_object_type eq $F{PERSON} && (! &handle_in_arin_db($handle, $check::meta  )  )  )  ) {
    
            &prob($hash_ref, "OBJECT_NOT_IN_DATABASES", $on_form_object_type, $handle);
        }
    
        &add_to_field(\$check::found{$F{NICHDL}}, $handle);
        #$check::found{$hash_ref->{$F{NICHDL}}}++;
    }

    # Looks like attempted AUTO NIC -> problem
	# LEE: 19980128: if they put AUTO1 and there's an AUTO1 nic-hdl object 
	# in the DB we don't get this far. Oops.
	# But if we move this up, we might match an actual NIC handle.

    elsif ( $handle =~ /$NEARLY_AUTONIC_REG/i) {
        &prob($hash_ref, "LOOKS_LIKE_AUTO_NIC_HANDLE", $F{NICHDL}, $handle);
    }

    # Otherwise -> problem (invalid value)    

    else {
        &prob($hash_ref, "INVALID_NIC_HANDLE_IN_PERSON_OBJECT", $F{NICHDL}, $handle );
    }
}

#   Purpose         :   Compare the things we need with the things
#                       we found, amongst other things
#   Side Effects    :   ?
#   Comments        :   
#   In              :   VOID
#   Out             :   VOID
#
sub check_post_processing_values {
    
    # Check for each template that minimum number of templates of this type was found

    my ($template_key, $template_name, $num_found_of_type, $min_no_templates_of_type);

    # For each template type (equivalent to: each template name)...

    foreach $template_key (keys %TEMPLATE_NAMES) {

        my $template_name = &template_name($template_key);
    
        # Extract minimum for this template type
    
        $min_no_templates_of_type = &template_min_number($template_key);
    
        # Extract how many were found 
        
        $num_found_of_type = $check::found{$template_name} || 0;
        
        # Fewer templates of this type found than needed -> problem
    
        &prob($check::meta, "TOO_FEW_TEMPLATES_OF_TYPE_X", $template_name, $min_no_templates_of_type)   
            if ( $num_found_of_type < $min_no_templates_of_type );
    }
    
    ### Check REQUEST OVERVIEW TEMPLATE against various fields in other templates ###
    
    # pi-req against status line in network template
    # We only compare if both are defined. If they're not
    # a problem will have been generated elsewhere to that effect.
    
    if ( ! &absent($check::found{$F{STATUS}}) && ! &absent($check::found{$F{PIREQ}}) ) {
        
        &prob($check::meta, "PI_MISMATCH_BETWEEN_OVERVIEW_AND_NETWORK_TEMPLATES") 
            if ( $check::found{$F{STATUS}} =~ /ASSIGNED PA/ && $check::found{$F{PIREQ}} ||
            $check::found{$F{STATUS}} =~ /ASSIGNED PI/ && !$check::found{$F{PIREQ}} );
    }

    # This text adds up details of the fields which have problems

    my $total_mismatch_text = "";

    ### ADDRESSES ###

    #request size against addressing plan
    $total_mismatch_text .= &cmp_totals( $F{RSIZE}, $check::found{$F{RSIZE}}, $F{ADRRSIZE}, $check::found{$F{ADRRSIZE}});

    # addresses immediate against addressing plan
    $total_mismatch_text .=  &cmp_totals( $F{OVRIMM}, $check::found{$F{OVRIMM}}, $F{ADRIMM}, $check::found{$F{ADRIMM}});

    # addresses-year-1 against addressing plan
    $total_mismatch_text .= &cmp_totals( $F{OVRYR1}, $check::found{$F{OVRYR1}}, $F{ADRYR1}, $check::found{$F{ADRYR1}});

    # addresses-year-2 against addressing plan
    $total_mismatch_text .= &cmp_totals( $F{OVRYR2}, $check::found{$F{OVRYR2}}, $F{ADRYR2}, $check::found{$F{ADRYR2}});

    # if there *were* any mismatches, give a problem with details of *all*
    # the problems in one

    &prob($check::meta, 'ADR_MISMATCH_WITH_REQUEST_OVERVIEW_TEMPLATE', "$total_mismatch_text")
        if ( ! &absent($total_mismatch_text) );

    ### SUBNETS ###

    $total_mismatch_text = "";

    # subnets totals against addressing plan
    $total_mismatch_text .=  &cmp_totals( $F{OVRSUBIMM}, $check::found{$F{OVRSUBIMM}}, $F{ADRSUBIMM}, $check::found{$F{ADRSUBIMM}});

    # subnets-year-1 against addressing plan
    $total_mismatch_text .= &cmp_totals( $F{OVRSUBYR1}, $check::found{$F{OVRSUBYR1}}, $F{ADRSUBYR1}, $check::found{$F{ADRSUBYR1}});

    # subnets-year-2 against addressing plan
    $total_mismatch_text .= &cmp_totals( $F{OVRSUBYR2}, $check::found{$F{OVRSUBYR2}}, $F{ADRSUBYR2}, $check::found{$F{ADRSUBYR2}});

    # if there *were* any mismatches, give a problem with details of *all*
    # the problems in one

    &prob($check::meta, 'SUB_MISMATCH_WITH_REQUEST_OVERVIEW_TEMPLATE', "$total_mismatch_text")
        if ( ! &absent($total_mismatch_text) );

    ### Check NETWORK TEMPLATE NIC handles from  can be found somewhere ###

    # It must either be on the form (person template) or as a person or 
    # role object in the RIPE or ARIN databases.

    if ( defined $check::need{$F{NICHDL}} ) {

        my $handle;
    
        my @handles_on_form = split $FORM_FS, $check::found{$F{NICHDL}}
            if (defined $check::found{$F{NICHDL}});
    
        foreach $handle (split $FORM_FS, $check::need{$F{NICHDL}}) {
    
            # We pass "ok" as a failure condition to 'indb'. This means that if 
            # 'whois' fails, the conditional will succeed ( anything but the empty
            # string or undefined is 'positive') and no problem report will be 
            # generated, other than that in 'indb' saying there's a problem with
            # the database.
    
            &prob($check::meta, "NIC_HANDLE_UNKNOWN", $handle)  

                if ( ( ! defined $check::found{$F{NICHDL}} || (scalar (grep { $_ eq $handle } @handles_on_form) < 1) ) &&
                    ! &indb("person", "ok", $handle, $check::meta) &&
                    ! &indb("role", "ok", $handle, $check::meta) &&
                    ! &handle_in_arin_db($handle, $check::meta) );
        }
    }
}

#   Purpose         :   Compare totals from the request overview and addressing
#                       plan templates. Return a string detailing the mismatch if
#                       there is one.
#   Side Effects    :   ?
#   Comments        :   
#   In              :   $: the fieldname in question from the overview template
#                       $: the value of that field in the overview template
#                       $: the fieldname in question from the address plan template
#                       $: the value of that field in the address plan template
#   Out             :   "" or details of the mismatch
#
sub cmp_totals {

    my $overview_fieldname = shift @_;
    my $overview_value = shift @_;
    my $addrplan_fieldname = shift @_;
    my $addrplan_value = shift @_;

    # Mismatch -> problem
    # If one of the values doesn't exist, we DON'T produce a mismatch error.
    # A problem will already have been generated when the template was checked.

    return "'$overview_fieldname' ($overview_value) and the number of $addrplan_fieldname ($addrplan_value) on the plan. "
        if ( ! &absent($overview_value) && ! &absent($addrplan_value) && $overview_value != $addrplan_value );

    return "";
}

#   Purpose  :  Add a problem code plus arguments to a template.
#   In       :  $%: pointer to the template to add a problem code to
#               $: the value of that field in the overview template
#               $: the fieldname in question from the address plan template
#               $: the value of that field in the address plan template
#   Out      :  VOID
#
sub prob {

    my $hash_ref = shift @_;

    my $arg;
    my $args = shift @_;


# MGTM don't show some probs if this is a web report
    if ( ! ( &show_this_prob( $args ) ) ) {
	return;
    }


    $args = "$args$PROB_FS$arg"
        while ( defined ($arg = shift @_) );

    # Add it to the PROBLEMS field in the template

    push @{ $hash_ref->{$F{PROBLEMS}} }, $args;

    &dprint($hash_ref->{$F{TNAME}}.": ".$args);
}


#   Purpose  :  Return 1 if this problem should be shown
#   In       :  $: problem type 
#   Out      :  $: 1 if should ignore else undef
#
sub show_this_prob
    {
    my ( $probType ) = @_;	# ARGS: problem type
	# RET : 1 to show, else undef

    if ( $REPORT_TYPE_FLAG == $REPORT_TYPE{HTML} )
# or is that: if ( $REPORT_TYPE_FLAG == $REPORT_TYPE{ $TYPE{HTML} } ) :-) ?
	# web report
	{
	if ( defined $DONT_REPORT_THESE_FOR_WEB{$probType} )
		# don't show it
		{
		return;
		}
	else
		# show it
		{
		return 1;
		}
	}
    else
	# other report type - show it
	{
	return 1;
	}

    }


#   Purpose  :  For a given template, attach problem codes if it has fields that
#        should a single value but do not.
#   In       :  $%: A pointer to the hash containing the template to be checked.
#   Out      :  VOID
#   Comments :  @F_MUST_BE_SINGEL_VALUE holds an array of field names which must have
#               a single value. This routine goes through this array, and if any of 
#               the items occur in the template as a field, but the field DOESN'T
#               have an single value, a problem code is attached to the template
#               and the rest of the field removed.
#
sub single_value_check {

    my $hash_ref = shift @_;

    my $field;

    # Go through each of the field names which must have a single value

    foreach $field (@F_MUST_BE_SINGLE_VALUE) {

        # Ignore it if this field name isn't defined in the template we're checking.
    
        if ( &absent( $hash_ref->{$F{$field}} ) ) {
            &prob( $hash_ref, "FIELD_IS_BLANK", $F{$field} );
            next;
        }
    
        # Find a value at start of field
    
        $hash_ref->{$F{$field}} =~ /^\s*(\S+)(.*)$/;
        
        my ($value, $after_value) = ($1, $2);
    
        # if there was something more than the value... 
        
        if ( $after_value =~ /\S/ ) {
                
            # amend the field to hold only the value we found
             
            $hash_ref->{$F{$field}} = $value;
    
            # and say we've done so
    
            &prob($hash_ref, "MUST_BE_A_SINGLE_VALUE", $F{$field}, $value, "$after_value");
        }
    }
}

#   Purpose  :  For a given template, attach problem codes if it has fields that
#        should hold integer values but do not.
#   In       :  $%: A pointer to the hash containing the template to be checked.
#   Out      :  VOID
#   Comments :  @F_MUST_BE_INTEGER_VALUE holds an array of field names which must have
#               an integer value. This routine goes through this array, and if any of
#               the items occur in the template as a field, but the field DOESN'T
#               have an integer value, a problem code is attached to the template and
#               the rest of the field removed.
#
sub integer_check {

    my $hash_ref = shift @_;

    my $field_name;

    my $value;

    # Go through each of the field names which must have a single value

    foreach $field_name (@F_MUST_BE_INTEGER) {

        $value = $hash_ref->{$F{$field_name}};
    
        # Ignore it if this field name isn't defined in the template we're checking.
    
        next 
            if ( &absent( $value ) );
    
        # If it's a slash notation and it's allowed to be..
    
        if ( $value =~ /^\/(\d+)/ && scalar( grep { $_ eq $field_name } @F_MAY_USE_SLASH_NOTATION) > 0 ) {
    
            # Convert it or signify a problem if we failed to
    
            &prob($hash_ref, 'CONVERTED_SLASH_TO_INT', $F{$field_name}, $hash_ref->{$F{$field_name}}, $value )
                if ( $value = &slash2int($value) );
        }
    
        # If we didn't end up with a value we can use, give an error

        if ( ! defined $value || $value !~ /^\s*\d+\s*$/ ) {
            &prob($hash_ref, "VALUE_MUST_BE_AN_INTEGER", $F{$field_name}, $hash_ref->{$F{$field_name}} );

            # We also delete the field, so the bad value isn't used elsewhere

            delete $hash_ref->{$F{$field_name}};
        }

        # Set the field to either the original or modified value, depending
        # on what happened above

        else {
            $hash_ref->{$F{$field_name}} = $value;
        }  
    }
}

#   Purpose  :  Foreach field in a given template, check whether it is allowed to
#               have multipe values, and record a problem in the template
#               if it does and is not allowed to :)
#   In       :  $%: A pointer to the hash containing the template to be checked.
#   Out      :  VOID
#   Comments :  @F_MULTIPLE_ALLOWED holds an array of field names which are allowed
#               to have more that one line on the request form.
#               This routine goes through each field in the template, and if any of
#               the field values have $FORM_FS in them (this is placed in the field
#               by Extract.pm to separate multiple lines for one field) but are not 
#               listed in @F_MULTIPLE_ALLOWED, a problem code is attached to the 
#               template.
#
sub multiple_check {

    my $hash_ref = shift @_;

    my ( $allowed, $field );

    # Go through each field in the template we're checking.

    foreach $field (keys %$hash_ref) {

        # if multiple lines were found in request for this field
    
        if ( $hash_ref->{$field} =~ /$FORM_FS/ ) {
    
			# Lee 980128: added this because it was warning:
			# 'rev-srv field can't have multiple lines'
			# If it's one of the fields in %F (we don't check
			# rev-srv lines for example: it's not important for
			# the request

            # and it wasn't found in @F_MULTIPLE_ALLOWED (i.e. multiple
            # values not allowed
    
            if ( scalar ( grep { $F{$_} eq $field } @F_MULTIPLE_ALLOWED) < 1
				&& scalar ( grep { $F{$_} eq $field } keys %F) > 0 ) {
    
                # then it shouldn't be a multiple field. We try to amend the
                # situation by removing all the excess data.
        
                # but first we store all of the values and replace $FS with
                # $VISIBLE_FS so that they can be displayed in the problem
                # report.
        
                my $all_values = $hash_ref->{$field};
                $all_values =~ s/$FORM_FS/$VISIBLE_FS/g;
                
                $hash_ref->{$field} =~ s/$FORM_FS(.*)$//g;
                
                # and we inform that we've done so
                &prob($hash_ref, "MULTIPLE_LINES_NOT_ALLOWED", $field, $hash_ref->{$field}, "$all_values");
            
            }
        }
    }
}

#   Purpose  :  Check a template to see if there may be more than one object
#               included under the template header.
#   In       :  $%: A pointer to the hash containing the template to be checked.
#   Out      :  VOID
#
# 
sub multiple_keyfield_check {

    my $hash_ref = shift @_;

    my $field_name;

    my ( $no_keyfields_found, $template_types_found ) = ( 0, undef );

    # For each field in the template to check...

    foreach $field_name ( keys %$hash_ref ) {

        # See if this is the key field of a template

        my $template_type = &is_template_keyfield($field_name);
    
        # If so, record the fact that we've found it

        if ( $template_type ) {
    
            # Record the total number of keyfields found

            $no_keyfields_found ++;  
    
            # Record the name of the template found

            if (defined $template_types_found) {
                $template_types_found .= ", $template_type";
            }
            else {
                $template_types_found = $template_type; 
            }

            # If there were multiple keys of this type found, add to the total
            # We don't add the amount found: it doesn't matter because all we're
            # going to check for later is if > 1 we're found

            $no_keyfields_found ++
                if ( $hash_ref->{$field_name} =~ $FORM_FS );
        }
    }

    &prob($hash_ref, 'MULTIPLE_KEY_FIELDS_FOUND', "$template_types_found")
        if ( $no_keyfields_found > 1 );
}

#   Purpose  :  Foreach field in a given template, check whether the field name
#               looks like it might be a misspelt version of an actual field name.
#               Record a problem if it does.
#   In       :  $%: A pointer to the hash containing the template to be checked.
#   Out      :  VOID
#   Comments :  %F_MISSPELLING has fields which represent misspellings. The key
#               represents a possible misspelling, the value represents what the
#               actual key should be. This routine goes through each field in the
#               template to check, and if any of the field names match one of the 
#               misspelt field names in %F_MISSPELLING, a problem is generated.
#
sub misspelling_check {

    my $hash_ref = shift @_;

    my $field_name;

    # For each field in the template to check...

    foreach $field_name (keys %$hash_ref) {

        # Problem reported if it's field name is the same as one of the misspelt
        # field names. The field name is ocnverted to the correct one.
    
        if ( defined $F_MISSPELLING{$field_name} ) {
            &prob($hash_ref, "MISSPELT_FIELD_NAME", $field_name, $F_MISSPELLING{$field_name});
            # Copy the value accross to the correct field name
            
            $hash_ref->{$F_MISSPELLING{$field_name}} = $hash_ref->{$field_name};
            
            # Delete the old field
            delete $hash_ref->{$field_name};

	    # Update the $F{FIELD_ORDER} field, which lists the
	    # old field
	    # The field order is reconstructed into @new_field_order,
	    # but the reference to the bad field name is replaced
	    # with the corrected version.

	    my ($old_field, @new_field_order);

	    foreach $old_field ( @{$hash_ref->{$F{FIELD_ORDER}}} ) {
		if ( $old_field eq $field_name ) {
		    push @new_field_order, $F_MISSPELLING{$field_name};
		}
		else {
		    push @new_field_order, $old_field;
		}
	    }
	    @ {$hash_ref->{$F{FIELD_ORDER}}} = @new_field_order;
        }
    }
}

#   Purpose  :  Provide a comparison function which sorts a list of alphanumeric
#               items, pure numeric items rank higher than non-pure-numeric.
#   In       :  VOID
#   Out      :  $:  boolean: see below for description.
#   Comments :  This would be better placed in in Misc.pm, but Perl stipulates that
#               sort functions must be in the same package as where they are used.
#               Example sort:
#               Before: ( 1, 6, ab, 10, a, 5, g, c )
#               After:  ( 1, 5, 6, 10, a, ab, c, g )
#
sub mixsort {
    
    # Examine both items to check whether they're numeric or non-numeric
        
    my $a_is_numeric_flag = ($a =~ /^\d+$/);
    my $b_is_numeric_flag = ($b =~ /^\d+$/);

    # If both numeric, do <=>

    return $a <=> $b 
        if ( $a_is_numeric_flag && $b_is_numeric_flag );

    # If both non-numeric, do cmp

    return $a cmp $b 
        if ( ! $a_is_numeric_flag && ! $b_is_numeric_flag );

    # In a comparison between numeric and non-numeric, numeric wins

    return -1 
        if ( $a_is_numeric_flag );  

    return 1; 
}

#   Purpose     :   Check a field for something that looks like
#                   a subnet line from a plan, but with a mistake.
#   In          :   $%: pointer to the hash we want to check.
#   Out         :   $: boolean: 0 = nothing found 1 = suspected subnets 
#                   found.
#
sub subnets_in_remainder {

    my $hash_ref = shift @_;

    my $match_found_flag = 0;

    my $remainder_line;

    # If there *isn't* any remainder text, return 'not found'
    return 0
        if ( ! defined $hash_ref->{$F{REMAINDER}} );

    foreach $remainder_line ( split /\n/, $hash_ref->{$F{REMAINDER}} ) {
    
        # Check for a line which would result if the subnet mask was omitted
    
        if ($remainder_line =~ /$MISSING_MASK_REG/) {
            &prob($hash_ref, "MISSING_PREFIX_OR_SUBNET_MASK", $remainder_line);
            $match_found_flag = 1;
        }
    
        # Check for a line which would result if one of the values was omitted
        
        elsif ($remainder_line =~ /$MISSING_VALUE_REG/) {
            &prob($hash_ref, "MISSING_SIZE_OR_USAGE_ESTIMATE", $remainder_line);
            $match_found_flag = 1;
        }
    }

    return $match_found_flag;
}

#   Purpose     :   Count the number of subnet lines in a plan.
#   In          :   $%: pointer to the hash we want to check.
#   Out         :   $: boolean: 0 = ok, 1 = plan too big
#
sub plan_too_big {

    my $hash_ref = shift @_;

    my $line;
    my $lines_found;

    $lines_found = scalar ( grep { $_ =~ /^\d+$/ } sort mixsort keys %$hash_ref );

    if ( $lines_found > $PLANLINES_IGNORE_THRESHOLD) {

        &prob($hash_ref, "PLAN_TOO_LARGE");
        return 1;
    }

    return 0;
}

#   Purpose     :   Generate a 'bad sunet mask' problem, but try to
#                   make suggestions too.
#   In          :   $%: pointer to the hash we want to check.
#                   $: size of subnet with incorrect mask.
#                   $: the incorrect mask
#                   $: the prefix of the subnet
#                   $: the line number of the subnet on the plan
#   Out         :   VOID
#
sub bad_subnet_mask_problem {

    my ($hash_ref, $size, $mask, $prefix, $line) = @_;

    # We assume to being with we can't make a suggestion.

    my $suggestion_text = " ";
    
    # Initial guess
    
    my $suggestion = &subnetsize2mask($size);
    
    # If initial guess didn't result a in valid mask,
    # try it with $size+2. They often quote the
    # size of the subnet as being the number of usable
    # addresses
    
    $suggestion = &subnetsize2mask($size + 2)
        if (! &ismask($suggestion));
    
    # If we got a valid mask with one of the above, suggest it
    
    $suggestion_text = "Suggest $suggestion for this subnet size ($size)."
        if ( &ismask($suggestion) );
    
    &prob($hash_ref, "INVALID_SUBNET_MASK", $mask, $prefix, $line, $suggestion_text); 
}

1;
