#include "cupsparserwindow.h"
#include "cupsparserview.h"
#include "cupsaccessparser.h"
#include "cupserrorparser.h"
#include "cupspageparser.h"
#include "parserdialog.h"

#include <qpopupmenu.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kapplication.h>
#include <klocale.h>
#include <kwin.h>
#include <kstatusbar.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kconfig.h>

CupsParserWindow::CupsParserWindow( WindowType type, QWidget *parent, const char *name )
	: KMainWindow( parent, name ), m_type( type )
{
	m_view = new CupsParserView( this );
	connect( m_view, SIGNAL( selectionChanged() ), SLOT( selectionChanged() ) );
	connect( m_view, SIGNAL( itemsChanged( int ) ), SLOT( itemsChanged( int ) ) );
	connect( m_view, SIGNAL( rightButtonPressed( QListViewItem*, const QPoint&, int ) ),
			SLOT( rightButtonPressed( QListViewItem*, const QPoint&, int ) ) );

	setCentralWidget( m_view );

	CupsParser *parser = 0;
	switch ( type )
	{
		case Access:
			parser = new CupsAccessParser;
			setCaption( "CUPS " + i18n( "Access log" ) );
			break;
		case Error:
			parser = new CupsErrorParser;
			setCaption( "CUPS " + i18n( "Error log" ) );
			break;
		case Page:
			parser = new CupsPageParser;
			setCaption( "CUPS " + i18n( "Page log" ) );
			break;
	}
	connect( parser, SIGNAL( error( const QString& ) ), SLOT( error( const QString& ) ) );
	m_view->setParser( parser );

	initActions();

	KConfig *conf = KGlobal::config();
	QSize dummy( 650, 500 );
	conf->setGroup( parser->parserName() );
	resize( conf->readSizeEntry( "Size", &dummy ) );
	m_view->slotGrouping( conf->readNumEntry( "Grouping", 0 ) );
	static_cast<KSelectAction*>( actionCollection()->action( "group_items" ) )->setCurrentItem( m_view->grouping() );
	m_view->setLimit( conf->readNumEntry( "ViewLimit", -1 ) );
	m_view->setRestored( conf->readNumEntry( "Restored", -1 ) != -1 );
}

CupsParserWindow::~CupsParserWindow()
{
	KConfig *conf = KGlobal::config();
	conf->setGroup( m_view->parser()->parserName() );
	conf->writeEntry( "Size", size() );
	conf->writeEntry( "Grouping", m_view->grouping() );
	conf->writeEntry( "ViewLimit", m_view->limit() );
	conf->writeEntry( "Restored", m_view->restored() ? type() : -1 );
}

void CupsParserWindow::initActions()
{
	KStdAction::quit( this, SLOT( closeAll() ), actionCollection() );
	KStdAction::close( this, SLOT( close() ), actionCollection() );

	KAction *a = 0;
	a = new KAction( KGuiItem( i18n( "&Details" ), "viewmag" ), CTRL+Key_D, m_view, SLOT( slotProperty() ), actionCollection(), "view_details" );
	a->setEnabled( false );
	a = new KAction( KGuiItem( i18n( "Resu&me parsing" ), "player_play" ), CTRL+Key_M, this, SLOT( resumeParsing() ), actionCollection(), "resume_parsing" );
	a->setEnabled( false );
	a = new KAction( KGuiItem( i18n( "&Stop parsing" ), "player_pause" ), CTRL+Key_S, this, SLOT( stopParsing() ), actionCollection(), "stop_parsing" );
	new KAction( KGuiItem( i18n( "&Restart parsing" ), "player_start" ), CTRL+Key_R, this,  SLOT( restartParsing() ), actionCollection(), "restart_parsing" );
	KSelectAction *sa = new KSelectAction( i18n( "Group by..." ), 0, actionCollection(), "group_items" );
	QStringList items;
	items << i18n( "None" ) << QString::null;
	for ( int i=0; i<m_view->columns(); i++ )
		items << m_view->columnText( i );
	sa->setItems( items );
	sa->setCurrentItem( 0 );
	connect( sa, SIGNAL( activated( int ) ), m_view, SLOT( slotGrouping( int ) ) );
	new KAction( i18n( "&Access log" ), CTRL+Key_A, this, SLOT( showWindow() ), actionCollection(), "view_access" );
	new KAction( i18n( "&Error log" ), CTRL+Key_E, this, SLOT( showWindow() ), actionCollection(), "view_error" );
	new KAction( i18n( "&Page log" ), CTRL+Key_P, this, SLOT( showWindow() ), actionCollection(), "view_page" );
	new KAction( KGuiItem( i18n( "Configure parser" ), "configure" ), CTRL+Key_O, this, SLOT( configure() ), actionCollection(), "configure_parser" );

	createGUI();

	KStatusBar *bar = statusBar();
	bar->insertFixedItem( "10000/10000", 0 );
	bar->changeItem( "", 0 );
	bar->insertFixedItem( "  " + i18n( "Parsing" ) + "  ", 1 );
}

void CupsParserWindow::stopParsing()
{
	actionCollection()->action( "resume_parsing" )->setEnabled( true );
	actionCollection()->action( "stop_parsing" )->setEnabled( false );
	statusBar()->changeItem( i18n( "Stopped" ), 1 );
	m_view->parser()->stopParsing();
}

void CupsParserWindow::resumeParsing()
{
	actionCollection()->action( "stop_parsing" )->setEnabled( true );
	actionCollection()->action( "resume_parsing" )->setEnabled( false );
	statusBar()->changeItem( i18n( "Parsing" ), 1 );
	m_view->parser()->resumeParsing();
}

void CupsParserWindow::restartParsing()
{
	actionCollection()->action( "stop_parsing" )->setEnabled( true );
	actionCollection()->action( "resume_parsing" )->setEnabled( false );
	m_view->parser()->startParsing();
}

void CupsParserWindow::selectionChanged()
{
	QListViewItem *item = m_view->selectedItem();
	KAction *a = actionCollection()->action( "view_details" );
	if ( a )
		a->setEnabled( item && item->childCount() == 0 );
}

void CupsParserWindow::configure()
{
	ParserDialog dlg( this );

	bool wasParsing = actionCollection()->action( "stop_parsing" )->isEnabled();
	stopParsing();
	dlg.setParser( m_view );
	if ( dlg.exec() )
	{
		dlg.updateParser( m_view );
		m_view->parser()->startParsing( true );
	}
	if ( wasParsing )
		resumeParsing();
}

CupsParserWindow::WindowType CupsParserWindow::type() const
{
	return m_type;
}

void CupsParserWindow::showWindow()
{
	QString wname( sender()->name() );
	WindowType t;

	if ( wname == "view_access" )
		t = Access;
	else if ( wname == "view_error" )
		t = Error;
	else if ( wname == "view_page" )
		t = Page;
	else
		return;

	QPtrListIterator<KMainWindow> it( *KMainWindow::memberList );
	for ( ; it.current(); ++it )
		if ( static_cast<CupsParserWindow*>( it.current() )->type() == t )
			break;
	if ( it.current() )
	{
		KWin::setActiveWindow( it.current()->winId() );
	}
	else
	{
		CupsParserWindow *w = new CupsParserWindow( t );
		w->show();
	}
}

void CupsParserWindow::itemsChanged( int n )
{
	QString s = QString::number( n );
	if ( m_view->limit() != -1 )
		s.append( "/" + QString::number( m_view->limit() ) );
	statusBar()->changeItem( s, 0 );
}

void CupsParserWindow::closeAll()
{
	QPtrListIterator<KMainWindow> it( *KMainWindow::memberList );
	for ( ; it.current(); ++it )
		if ( it.current() != this )
			it.current()->close();
	close();
}

void CupsParserWindow::rightButtonPressed( QListViewItem *item, const QPoint& pos, int )
{
	QPopupMenu menu( this );
	actionCollection()->action( "group_items" )->plug( &menu );
	if ( item && item->childCount() == 0 )
	{
		menu.insertSeparator();
		actionCollection()->action( "view_details" )->plug( &menu );
	}
	m_view->parser()->setupRMB( item, &menu );
	menu.exec( pos );
}

void CupsParserWindow::error( const QString& msg )
{
	actionCollection()->action( "resume_parsing" )->setEnabled( false );
	actionCollection()->action( "stop_parsing" )->setEnabled( false );
	statusBar()->changeItem( i18n( "Stopped" ), 1 );
	KMessageBox::error( this, msg );
}

void CupsParserWindow::restoreWindows()
{
	QStringList parsers = CupsParser::parsers();
	KConfig *conf = KGlobal::config();
	CupsParserWindow *w = 0;
	for ( QStringList::ConstIterator it=parsers.begin(); it!=parsers.end(); ++it )
	{
		conf->setGroup( *it );
		int restoreFlag = conf->readNumEntry( "Restored", -1 );
		if ( restoreFlag != -1 )
		{
			w = new CupsParserWindow( ( CupsParserWindow::WindowType )restoreFlag );
			w->show();
		}
	}

	if ( w == 0 )
	{
		w = new CupsParserWindow( CupsParserWindow::Error );
		w->m_view->setRestored( true );
		w->show();
	}
}

#include "cupsparserwindow.moc"
