/*****************************************************************************
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 ****************************************************************************/

package org.apache.padaf.preflight;

import org.apache.padaf.preflight.javacc.ParseException;

/**
 * A PdfParseException is thrown when the JavaCC Parser can't validate the pdf
 * file. In this case a PdfParseException (or an inherited class) is thrown.
 * 
 * A PdfParseException inherits from ParseException to avoid compilation errors
 * in the classes generated by JavaCC.
 */
public class PdfParseException extends ParseException {
  protected boolean isTokenMgrError = false;
  protected String errorCode = null;
  protected int line = 0;

  /**
   * This constructor clones the given ParseException and initialize the
   * errorCode if it is possible. (e is an instance of PdfParseException)
   * 
   * @param e
   */
  public PdfParseException(ParseException e) {
    super();
    this.currentToken = e.currentToken;
    this.expectedTokenSequences = e.expectedTokenSequences;
    this.tokenImage = e.tokenImage;
    this.initCause(e);
    if (e instanceof PdfParseException) {
      this.errorCode = ((PdfParseException) e).errorCode;
    }
  }

  /**
   * This constructor calls the PdfParseException(String message, String code)
   * constructor with a code set to null.
   * 
   * @param message
   *          the explanation message (The message of TokenMngError). This field
   *          is mandatory.
   */
  public PdfParseException(String message) {
    this(message, null);
  }

  /**
   * This constructor is generally used when the PDF validation fails due to a
   * TokenMngError. In this case, the message should be the message of the
   * TokenMngError.
   * 
   * @param message
   *          the explanation message (The message of TokenMngError). This field
   *          is mandatory.
   * @param code
   *          the error code if it can be determined by the creator of this
   *          exception (Can be null)
   */
  public PdfParseException(String message, String code) {
    super(message);
    this.isTokenMgrError = true;
    int lineIndex = message.indexOf("Lexical error at line ");
    if (lineIndex > -1) {
      String truncMsg = message.replace("Lexical error at line ", "");
      String nbLine = truncMsg.substring(0, truncMsg.indexOf(","));
      line = Integer.parseInt(nbLine);
    }
    this.errorCode = code;
  }

  /**
   * Get the validation error code
   * 
   * @return
   */
  public String getErrorCode() {
    return this.errorCode;
  }
}
