/*=============================================================================
 * Copyright (c) 1998-1999 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission. For written permission, please contact
 *    apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *=============================================================================
 *
 * NAME
 *	EAccess - Extended Access control module
 *
 * AUTHOR
 *	Patrick Asty <pasty@micronet.fr>
 *
 * VERSION
 *	$Revision: 2.2 $
 *
 * DOCUMENTATION
 *	See doc/index.html...
 *
 * CHANGELOG
 *	$Log: mod_eaccess.c,v $
 *	Revision 2.2  1999/06/26 04:31:08  pasty
 *	auth/securid OK
 *
 *	Revision 2.1  1999/06/24 06:20:37  pasty
 *	2.0.1.4 -> 2.1
 *
 *	Revision 2.0.1.4  1999/06/17 09:42:36  pasty
 *	Fixed: no cache creation if EAccessEnable = off
 *
 *	Revision 2.0.1.3  1999/06/17 09:37:08  pasty
 *	Fixed glibc 2.1 ndbm.h inclusion problems.
 *
 *	Revision 2.0.1.2  1999/06/15 19:40:53  pasty
 *	+ clean
 *
 *	Revision 2.0.1.1  1999/06/09 21:46:24  pasty
 *	doc
 *
 *	Revision 2.0.1.0  1999/06/09 14:14:42  pasty
 *	auth/basic done.
 *
 *	Revision 2.0  1999/06/07 19:31:22  pasty
 *	add permit/deny/warning on EAccessRule, auth not implemented
 *
 *	Revision 1.6  1999/06/05 06:30:08  pasty
 *	Don't create logfile if loglevel == 0.
 *	Correction: all methods may have arguments and body.
 *
 *	Revision 1.5  1999/06/02 05:19:07  pasty
 *	Add logging
 *
 *	Revision 1.4  1999/05/30 14:05:17  pasty
 *	add blah blah
 *
 *	Revision 1.3  1999/05/29 07:45:44  pasty
 *	POST (and PUT) method done
 *
 *	Revision 1.2  1999/05/29 05:15:55  pasty
 *	GET method OK.
 *	todo: POST method...
 *
 *	Revision 1.1  1999/05/28 04:25:43  pasty
 *	Initial revision
 *
 *=============================================================================
 */
/* static char rcsid [] = "$Id: mod_eaccess.c,v 2.2 1999/06/26 04:31:08 pasty Exp $"; */

#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "http_log.h"
#include "util_script.h"
#include "http_main.h"
#include "http_request.h"
#include "http_core.h"
#include "util_md5.h"
#if defined (__GLIBC__)						&&	\
    defined (__GLIBC_MINOR__)					&&	\
    __GLIBC__ >= 2						&&	\
    __GLIBC_MINOR__ >= 1
#include <db1/ndbm.h>
#else
#include <ndbm.h>
#endif

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE !(FALSE)
#endif

/*
 * Dfinitions pour la compilation du module: Configure utilise le texte situ
 * entre -START et -END.
 *
 * MODULE-DEFINITION-START
 * Name: eaccess_module
 * ConfigStart
    . ./helpers/find-dbm-lib
 * ConfigEnd
 * MODULE-DEFINITION-END
 */

/*
 * Pr-dclaration du module pour les cross-references
 */
module eaccess_module;

/*
 * La config (globale) du module
 */
#define EACCESS_DISABLED	1<<0
#define EACCESS_ENABLED		1<<1

/*
 * Les actions possibles dans une rgle
 */
typedef enum
{
  EACCESS_DENY,				/* deny				*/
  EACCESS_PERMIT,			/* permit			*/
  EACCESS_WARNING,			/* warning			*/
  EACCESS_AUTH_BASIC,			/* auth/basic[=n]		*/
  EACCESS_AUTH_SECURID			/* auth/securid[=n]		*/
} eaccess_action;

/*
 * Les actions en chaine de caractres
 */
#define EACCESS_DENY_STR		"deny"
#define EACCESS_PERMIT_STR		"permit"
#define EACCESS_WARNING_STR		"warning"
#define EACCESS_AUTH_BASIC_STR		"auth/basic"
#define EACCESS_AUTH_BASIC_LEN		10	/* strlen ("auth/basic")*/
#define EACCESS_AUTH_SECURID_STR	"auth/securid"
#define EACCESS_AUTH_SECURID_LEN	12	/* strlen ("auth/securid")*/

/*
 * Une rgle
 */
typedef struct eaccess_rulentry
{
  char			*pattern;	/* l'er en clair (avec le '!'	*/
  					/* ventuel			*/
  regex_t		*regexp;	/* l'ER compile		*/
  int			revert;		/* revert-match ('!')		*/
  eaccess_action	action;		/* l'action  entreprendre si	*/
  					/* l'ER match l'URL		*/
  char			*auth_opt;	/* les options des rgles	*/
  					/* auth/<*>			*/
  int			auth_ttl;	/* le TTL de l'auth, qd action	*/
  					/* == auth/<*>			*/
} eaccess_rulentry;

/*
 * La config du module (la config dynamique n'est pas implmente...)
 */
typedef struct eaccess_cfg
{
  int			state;		/* eaccess activ ou non	*/
  array_header		*rules;		/* tableau d'eaccess_rulentry	*/
  char			*logfile;	/* le fichier de log		*/
  int			logfd;		/* le file descriptor du log	*/
  int			loglevel;	/* la verbosit des logs	*/
  char			*cachefile;	/* le fichier de cache des auth	*/
  char			*cachefname;	/* le nom complet du cache auth	*/
} eaccess_cfg;


/*
 * mode_t pour les open()
 */
#ifdef WIN32
# define EACCESS_OPEN_MODE	(_S_IREAD | _S_IWRITE)
#else
# define EACCESS_OPEN_MODE	(S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH)
#endif

/*
 ************************************************************************
 * File locking and log time (cf mod_rewrite.c)
 ************************************************************************
 *
 * File locking and log time (cf mod_rewrite.c)
 */

#ifdef USE_FCNTL
static struct flock   lock_it;
static struct flock unlock_it;
#endif

static void fd_lock(request_rec *r, int fd)
{
    int rc = 0;

#ifdef USE_FCNTL
    lock_it.l_whence = SEEK_SET; /* from current point */
    lock_it.l_start  = 0;        /* -"- */
    lock_it.l_len    = 0;        /* until end of file */
    lock_it.l_type   = F_WRLCK;  /* set exclusive/write lock */
    lock_it.l_pid    = 0;        /* pid not actually interesting */

    while (   ((rc = fcntl(fd, F_SETLKW, &lock_it)) < 0)
              && (errno == EINTR)                               ) {
        continue;
    }
#endif
#ifdef USE_FLOCK
    while (   ((rc = flock(fd, LOCK_EX)) < 0)
              && (errno == EINTR)               ) {
        continue;
    }
#endif
#ifdef USE_LOCKING
    /* Lock the first byte, always, assume we want to append
       and seek to the end afterwards */
    lseek(fd, 0, SEEK_SET);
    rc = _locking(fd, _LK_LOCK, 1);
    lseek(fd, 0, SEEK_END);
#endif

    if (rc < 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                     "EAccess: failed to lock file descriptor");
        exit(1);
    }
    return;
}

static void fd_unlock(request_rec *r, int fd)
{
    int rc = 0;

#ifdef USE_FCNTL
    unlock_it.l_whence = SEEK_SET; /* from current point */
    unlock_it.l_start  = 0;        /* -"- */
    unlock_it.l_len    = 0;        /* until end of file */
    unlock_it.l_type   = F_UNLCK;  /* unlock */
    unlock_it.l_pid    = 0;        /* pid not actually interesting */

    rc = fcntl(fd, F_SETLKW, &unlock_it);
#endif
#ifdef USE_FLOCK
    rc = flock(fd, LOCK_UN);
#endif
#ifdef USE_LOCKING
    lseek(fd, 0, SEEK_SET);
    rc = _locking(fd, _LK_UNLCK, 1);
    lseek(fd, 0, SEEK_END);
#endif

    if (rc < 0) {
        ap_log_rerror(APLOG_MARK, APLOG_ERR, r,
                     "EAccess: failed to unlock file descriptor");
        exit(1);
    }
}

static char *current_logtime(request_rec *r)
{
    int timz;
    struct tm *t;
    char tstr[80];
    char sign;

    t = ap_get_gmtoff(&timz);
    sign = (timz < 0 ? '-' : '+');
    if (timz < 0) {
        timz = -timz;
    }

    strftime(tstr, 80, "[%d/%b/%Y:%H:%M:%S ", t);
    ap_snprintf(tstr + strlen(tstr), 80-strlen(tstr), "%c%.2d%.2d]",
                sign, timz/60, timz%60);
    return ap_pstrdup(r->pool, tstr);
}

/*
 ************************************************************************
 * Logger un message d'un niveau
 ************************************************************************
 */
static void eaccess_log (request_rec *r, int level, const char *text, ...)
{
  /*
   * La config actuelle
   */
  eaccess_cfg	*conf =
    (eaccess_cfg *) ap_get_module_config (r->server->module_config,
                                          &eaccess_module);
  /*
   * La liste du text, ... pass
   */
  va_list	ap;

  /*
   * Les "REMOTE" variables utilises pour le log
   */
  char		*ruser;
  const char	*rhost;

  /*
   * Les diffrentes parties de la ligne  logger
   */
  char		str1 [512];	/* l'interprtation de texte, ...	*/
  char		str2 [1024];	/* l'ensemble				*/

  /*
   * On rcupre la liste texte, ... passe en arguments
   */
  va_start (ap, text);

  /*
   * - si on n'a pas de file descripteur,
   * - ou si le message  logger est d'un niveau supprieur  celui demand par
   *   la directive LogLevel,
   * alors ce n'est pas la peine d'aller plus loin.
   */
  if ((conf->logfd < 0) || (level > conf->loglevel))
  {
    return;
  }

  /*
   * REMOTE_USER dispo?
   */
  if (r->connection->user == NULL)
  {
    ruser = "-";
  }
  else if (strlen (r->connection->user) != 0)
  {
    ruser = r->connection->user;
  }
  else
  {
    ruser = "\"\"";
  }

  /*
   * REMOTE_HOST
   */
  rhost = ap_get_remote_host (r->connection, r->server->module_config,
                              REMOTE_NOLOOKUP);
  if (rhost == NULL)
  {
    rhost = "UNKNOWN-HOST";
  }

  /*
   * La ligne  logger
   */
  ap_vsnprintf (str1, sizeof (str1), text, ap);
  ap_snprintf (str2, sizeof (str2), "%s %s %s %s %s\n",
	       rhost,
	       (
		 r->connection->remote_logname != NULL
		 ? r->connection->remote_logname
		 : "-"
	       ),
	       ruser,
	       current_logtime (r),
	       str1);

  /*
   * On crit
   */
  fd_lock (r, conf->logfd);
  write (conf->logfd, str2, strlen (str2));
  fd_unlock (r, conf->logfd);

  /*
   * On libre la liste et c'est fini
   */
  va_end (ap);
  return;
}

/*
 ************************************************************************
 * Stockage en cache des auth.
 ************************************************************************
 */
DBM *eaccess_auth_open (char *fname, int flags)
{
  DBM		*cachefd;

  cachefd = dbm_open (fname, flags, EACCESS_OPEN_MODE);
  return cachefd;
}

void eaccess_auth_close (DBM *fdesc)
{
  dbm_close (fdesc);
}

time_t eaccess_auth_get (request_rec *r,
                         char *cachefname, const char *auth)
{
  datum		key;
  datum		val;
  time_t	res;
  DBM		*cachefd;
  AP_MD5_CTX	context;
  
  if ((cachefd = eaccess_auth_open (cachefname, O_RDONLY)) == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "EAccess: could not open EAccessCache file '%s'",
		   cachefname);
    return (time_t) 0;
  }

  ap_MD5Init (&context);
  ap_MD5Update (&context, auth, strlen (auth));

  key.dptr  = ap_md5contextTo64 (r->pool, &context);
  key.dsize = strlen (key.dptr);
  val       = dbm_fetch (cachefd, key);
  if (val.dptr != NULL)
  {
    memcpy (&res, val.dptr, sizeof (time_t));
    eaccess_log (r, 2, "DB-GET: '%s' is found: time_t = %ld", auth, res);
    eaccess_auth_close (cachefd);
    return res;
  }
  eaccess_log (r, 2, "DB-GET: '%s' is NOT found", auth);
  eaccess_auth_close (cachefd);
  return (time_t) 0;
}

int eaccess_auth_put (request_rec *r,
                      char *cachefname, const char *auth, time_t *t)
{
  datum		key;
  datum		val;
  DBM		*cachefd;
  AP_MD5_CTX	context;
  
  if ((cachefd =
         eaccess_auth_open (cachefname, O_RDWR | O_APPEND)) == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "EAccess: could not open EAccessCache file '%s'",
		   cachefname);
    return (time_t) 0;
  }

  ap_MD5Init (&context);
  ap_MD5Update (&context, auth, strlen (auth));

  key.dptr  = ap_md5contextTo64 (r->pool, &context);
  key.dsize = strlen (key.dptr);
  val.dptr  = (void *) t;
  val.dsize = sizeof (time_t);
  dbm_store (cachefd, key, val, DBM_REPLACE);
  eaccess_log (r, 2, "DB-PUT: '%s' is stored", auth);
  eaccess_auth_close (cachefd);
  return 1;
}

int eaccess_auth_del (request_rec *r,
                      char *cachefname, const char *auth)
{
  datum		key;
  DBM		*cachefd;
  AP_MD5_CTX	context;
  
  if ((cachefd = eaccess_auth_open (cachefname, O_WRONLY)) == NULL)
  {
    ap_log_rerror (APLOG_MARK, APLOG_ERR, r,
                   "EAccess: could not open EAccessCache file '%s'",
		   cachefname);
    return (time_t) 0;
  }

  ap_MD5Init (&context);
  ap_MD5Update (&context, auth, strlen (auth));
  key.dptr  = ap_md5contextTo64 (r->pool, &context);
  key.dsize = strlen (key.dptr);
  dbm_delete (cachefd, key);
  eaccess_log (r, 2, "AUTH-DB: '%s' is deleted", auth);
  eaccess_auth_close (cachefd);
  return 1;
}

/*
 ************************************************************************
 * Initialisation du module (ouverture du log et de log des auth).
 ************************************************************************
 */
static void eaccess_init (server_rec *s, pool *p)
{
  /*
   * flags pour open()
   */
  int		flags_log   = (O_WRONLY | O_APPEND | O_CREAT);
  /*
   * La config actuelle
   */
  eaccess_cfg	*conf =
    (eaccess_cfg *) ap_get_module_config (s->module_config, &eaccess_module);
  /*
   * Le chemin d'accs complet au fichier de log
   */
  char		*logfname;
  /*
   * Pour rinitialiser le cache
   */
  DBM		*cachefd;

  /*
   * Si EAccessEnable n'est  on, on n'a pas grand chose  faire...
   */
  if (conf->state == EACCESS_DISABLED)
  {
    return;
  }

  /*
   * Si l'utilisateur n'a pas utilis de directive EAccessAuthCache, on utilise
   * la valeur par dfaut pour le fichier de cache
   */
  if (conf->cachefile == NULL)
  {
    conf->cachefile = "logs/eaccess_auth";
  }

  /*
   * On demande  complter par "Server Root" le nom du fichier si celui-ci
   * ne commence pas par un "/".
   */
  conf->cachefname = ap_server_root_relative (p, conf->cachefile);

  /*
   * On rinitialise le cache...
   */
  if ((cachefd = eaccess_auth_open (conf->cachefname,
			           O_WRONLY | O_CREAT | O_TRUNC)) == NULL)
  {
    ap_log_error (APLOG_MARK, APLOG_ERR, s,
                  "EAccess: could not create EAccessCache file '%s'",
		  conf->cachefname);
    exit (1);
  }
  eaccess_auth_close (cachefd);

  /*
   * Si EAccessLogLevel est  0, on n'a rien  faire...
   */
  if (conf->loglevel == 0)
  {
    return;
  }

  /*
   * Si l'utilisateur n'a pas utilis de directive EAccessLog, on utilise la
   * valeur par dfaut pour le fichier de log
   */
  if (conf->logfile == NULL)
  {
    conf->logfile = "logs/eaccess_log";
  }

  /*
   * On demande  complter par "Server Root" le nom du fichier si celui-ci
   * ne commence pas par un "/".
   */
  logfname = ap_server_root_relative (p, conf->logfile);

  /*
   * On ouvre...
   */
  if ((conf->logfd = ap_popenf (p, logfname, flags_log, EACCESS_OPEN_MODE)) < 0)
  {
    ap_log_error (APLOG_MARK, APLOG_ERR, s,
                  "EAccess: could not open EAccessLog file '%s'", logfname);
    exit (1);
  }

  /*
   * Et tout va bien
   */
  return;
}

/*
 ************************************************************************
 * Rcuprer l'"auth" dans une header et la supprimer
 ************************************************************************
 */
const char *eaccess_get_auth_basic (request_rec *r)
{
  /*
   * Rappelons le format d'une auth/basic dans un header HTTP de requte:
   *   Authorization: Basic toto:titi
   */
  const char	*auth = NULL;

  if ((auth = ap_table_get (r->headers_in, "Authorization")))
  {
    auth += strlen ("Basic ");
  }
  return (auth);
}

void eaccess_unset_auth_basic (request_rec *r)
{
  ap_table_unset (r->headers_in, "Authorization");
}

const char *eaccess_get_auth_securid (request_rec *r)
{
  /*
   * Rappelons le format d'un cookie dans un header HTTP de requte:
   *   Cookie: NAME1=OPAQUE_STRING1; NAME2=OPAQUE_STRING2 ...
   *
   * Pour SecurID, cela donne:
   *   Cookie: AceHandle=...; webid2=...
   */
  const char	*auth   = NULL;
  const char	*cookie = NULL;
  char		*value  = NULL;

  if ((cookie = ap_table_get (r->headers_in, "Cookie")))
  {
    if ((value = strstr (cookie, "AceHandle=")))
    {
      char	*end;

      value += strlen ("AceHandle=");
      auth   = ap_pstrdup (r->pool, value);
      end    = strchr (auth, ';');
      if (end)
      {
        *end = '\0';
      }
    }
  }
  return (auth);
}

void eaccess_unset_auth_securid (request_rec *r)
{
  /*
   * Un peu violent...
   */
  ap_table_unset (r->headers_in, "Cookie");
}

/*
 ************************************************************************
 * Code du vrificateur d'URL
 ************************************************************************
 */
static int eaccess_check (request_rec *r)
{
  /*
   * La config actuelle
   */
  eaccess_cfg		*conf =
    (eaccess_cfg *) ap_get_module_config (r->server->module_config,
                                          &eaccess_module);
  /*
   * == 1 si "Pattern"  match l'URL (ou si "!Pattern" n'a pas match l'URL)
   */
  int			matched;
  /*
   * Une  une, les ER de la liste des rgles de la config
   */
  eaccess_rulentry	*entries = (eaccess_rulentry *) conf->rules->elts;
  /*
   * Pour parcourir les ER
   */
  int			i;
  /*
   * l'"URL"  vrifier et ses data ventuelles (ou son body pour les POST/PUT).
   */
  char			url [HUGE_STRING_LEN];
  /*
   * 2 paramtres inutiliss pour regexec()
   */
  size_t		dummy_nmatch = 0;
  regmatch_t		dummy_pmatch [1];
  /*
   * Pour voir si un body est prsent, on en lira 1 caractre
   */
  char			c;

  /*
   * Si EAccessEnable n'est pas ON, on s'arrte l...
   */
  if (conf->state == EACCESS_DISABLED)
  {
    eaccess_log (r, 2, "EAccessEnable Off...");
    return OK;
  }

  /*
   * On "calcule" "<METHOD> <URI>"
   */
  ap_snprintf (url, sizeof (url), "%s %s", r->method, r->uri);

  /*
   * Si des arguments sont prsents, on ajoute "?<QUERY_STRING>"
   */
  if (r->args)
  {
    ap_snprintf (url, sizeof (url), "%s?%s", url, r->args);
  }

  /*
   * Si un body est prsent, on en ajoute le contenu "|<DATA>"
   *
   * Pour cela, on appelle ap_blookc() pour lire (sans le consommer) le prochain
   * caractre du body.
   * ap_blookc() a pour effet de remplir le buffer inptr si ce n'est pas dj
   * fait mais on doit vrifier qu'il y a quelque chose  lire sinon ap_blookc()
   * attend que le butineur fournisse des donnes...
   * ap_blookc retourne 1 si tout va bien.
   */
  if ((r->connection->client->incnt > 0) &&
      (ap_blookc (&c, r->connection->client) == 1))
  {
    int	len;
    /*
     * Ok, un body est diponible, on en ajoute le dbut  l'"url"  vrifier,
     * sous la forme: "POST url|dbut du body".
     *
     * 2 possibilits:
     *   - soit le body est complet dans le buffer, auquel cas il faut parfois
     *     supprimer les 2 derniers caractres CR-LF (?: cela dpend du
     *     butineur?),
     *   - soit le body n'est pas complet dans le buffer, auquel cas on garde
     *     tout ce qu'on a.
     *
     * Pour infos, url fait HUGE_STRING_LEN caractres de long (actuellement
     * 8192 dans httpd.h) et inptr DEFAULT_BUFSIZE (4096 dans buff.c).
     */
    len = strlen (url);
    ap_snprintf (url, sizeof (url), "%s|%s",
		 url, r->connection->client->inptr);
    /*
     * +1 pour le '|'
     */
    len += r->connection->client->incnt + 1;

    /*
     * Si les 2 derniers caractres sont CR-LF, on les supprime
     */
    if (url [len - 2] == '\r' && url [len - 1] == '\n')
    {
      len -= 2;
    }

    /*
     * Et on met fin  la chaine
     */
    url [len] = '\0';
  }

  /*
   * On parcourt les ER
   */
  for (i = 0; i < conf->rules->nelts; i++)
  {
    /*
     * On regarde si "<METHOD> <URI>" vrifie l'ER
     */
    matched = 0;
    if (regexec (entries[i].regexp, url, dummy_nmatch, dummy_pmatch, 0) == 0)
    {
      matched = 1;
    }

    /*
     * Si on est en revert-match, on inverse le rsultat
     */
    if (entries[i].revert)
    {
      matched = (matched) ? 0 : 1;
    }

    eaccess_log (r, 2, "RE '%s' %s URL '%s'", entries[i].pattern,
                 (matched) ? "matches" : "does not match", url);

    /*
     * Si l'ER matche, on dispatche suivant l'action  entreprendre
     */
    if (matched)
    {
      switch (entries[i].action)
      {
        case EACCESS_PERMIT:
	{
	  eaccess_log (r, 1, "RE #%03d grants access to '%s'", i + 1, url);
	  return OK;
	}
        case EACCESS_DENY:
	{
	  eaccess_log (r, 1, "RE #%03d denies access to '%s'", i + 1, url);
	  return FORBIDDEN;
	}
        case EACCESS_WARNING:
	{
	  eaccess_log (r, 1, "RE #%03d *** WARNING! *** '%s'", i + 1, url);
	  return OK;
	}
        case EACCESS_AUTH_BASIC:
        case EACCESS_AUTH_SECURID:
	{
	  /*
	   * L'ventuelle authentification dans le header, et l'ventuelle
	   * premire fois qu'elle a t utilise (en char* car dans une table
	   * on ne peut mettre que des char*; on convertira en time_t...)
	   */
	  const char	*auth;			/* user:pass en base64	*/
	  time_t	first_time;
	  const char	*auth_type;		/* le type de l'auth	*/
	  					/* pour les logs	*/
	  const char	*auth_opt;		/* le type de l'option	*/
	  					/* pour les logs	*/

	  if (entries[i].action == EACCESS_AUTH_BASIC)
	  {
	    auth_type = "basic";
	    auth_opt  = "realm";
	  }
	  else if (entries[i].action == EACCESS_AUTH_SECURID)
	  {
	    auth_type = "securid";
	    auth_opt  = "redirect";
	  }
	  else
	  {
	    auth_type = "<INTERNAL ERROR>";
	    auth_opt  = "<INTERNAL ERROR>";
	  }

	  eaccess_log (r, 2, "RE #%03d auth/%s: %s='%s', TTL=%d",
	               i + 1, auth_type, auth_opt,
		       entries[i].auth_opt, entries[i].auth_ttl);

	  /*
	   * Y a-t-il dans le Header une authentification?
	   */
	  if (((entries[i].action == EACCESS_AUTH_BASIC) &&
	       (auth = eaccess_get_auth_basic (r))) ||
	      ((entries[i].action == EACCESS_AUTH_SECURID) &&
	       (auth = eaccess_get_auth_securid (r))))
	  {
	    /*
	     * Il y a une authentification dans le Header HTTP.
	     */
	    eaccess_log (r, 2, "RE #%03d auth/%s: Authorization='%s'",
			 i + 1, auth_type, auth);
	    /*
	     * Cette authentification est-elle dj prsente dans la table des
	     * auth de cette ER?
	     */
	    if ((first_time = eaccess_auth_get (r, conf->cachefname, auth)))
	    {
	      /*
	       * Cette auth est dj connue, on va regarder si elle a expir.
	       * Donc avant tout, on regarde si TTL == 0 (=> pas d'expiration)
	       */
	      if (entries[i].auth_ttl)
	      {
		/*
		 * Ok, on s'intresse  l'expiration de cette authentification:
		 * le time_t de maintenant et le delta entre les 2 temps
		 */
		time_t	this_time;
		int	diff_time;

		/*
		 * La temps maintenant et la premire fois => delta
		 */
		time (&this_time);
		diff_time = this_time - first_time;
		eaccess_log (r, 2,
			     "RE #%03d auth/%s: Authorization already used: "
			     "first time=%ld => delta = %d seconds",
			     i + 1, auth_type, first_time, diff_time);

		/*
		 * Expire ou non?
		 */
		if (diff_time > entries[i].auth_ttl)
		{
		  /*
		   * L'authentification prsente dans le header est donc
		   * expire. Si la rgle avait prcis une option (cad un
		   * realm pour auth/basic, une redirection pour auth/securid),
		   * on dit 401 (auth/basic) ou 302 (auth/securid), sinon, on
		   * supprime l'auth dans le header et on continue la boucle...
		   */
		  if (entries[i].auth_opt)
		  {
		    if (entries[i].action == EACCESS_AUTH_BASIC)
		    {
		      /*
		       * auth/basic:
		       *
		       * Ok, la rgle prcise un realm: on supprime cette auth
		       * de la table et on retourne 401 au butineur...
		       */
		      eaccess_log (r, 2,
				   "RE #%03d auth/basic: Realm set to '%s' => "
				   "err 401 returned",
				   i + 1, entries[i].auth_opt);
		      eaccess_auth_del (r, conf->cachefname, auth);
		      ap_table_setn (r->err_headers_out, "WWW-Authenticate",
				     ap_pstrcat (r->pool, "Basic realm=\"",
				     ap_escape_quotes (r->pool,
						       entries[i].auth_opt),
				     "\"", NULL));
		      eaccess_log (r, 1, "RE #%03d AUTH too old for '%s'",
				   i + 1, url);
		      r->proxyreq = 0;
		      return AUTH_REQUIRED;
		    }
		    else if (entries[i].action == EACCESS_AUTH_SECURID)
		    {
		      /*
		       * auth/securid:
		       *
		       * Ok, la rgle prcise un realm: on supprime cette auth
		       * de la table, on crit "Location: <la redirection>" dans
		       * le header et on retourne 302 au butineur...
		       */
		      eaccess_log (r, 2,
				   "RE #%03d auth/securid: "
				   "Option set to '%s' => "
				   "redirected",
				   i + 1, entries[i].auth_opt);
		      eaccess_auth_del (r, conf->cachefname, auth);
		      ap_table_setn (r->headers_out, "Location",
				     ap_pstrdup (r->pool, entries[i].auth_opt));
		      eaccess_log (r, 1, "RE #%03d AUTH too old for '%s'",
				   i + 1, url);
		      /*r->proxyreq = 0;*/
		      return REDIRECT;
		    }
		    else
		    {
		      /*
		       * ... erreur dans le code ...
		       */
		      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
				   "EAccess: internal error: auth/???");
		      return SERVER_ERROR;
		    }
		  }
		  else
		  {
		    /*
		     * La rgle ne prcise pas d'option: on supprime cette auth
		     * du Header HTTP et de la table et on continue...
		     */
		    eaccess_log (r, 2,
				 "RE #%03d auth/%s: %s unset => auth cancelled",
				 i + 1, auth_type, auth_opt);
		    eaccess_auth_del (r, conf->cachefname, auth);
		    if (entries[i].action == EACCESS_AUTH_BASIC)
		    {
		      eaccess_unset_auth_basic (r);
		    }
		    else if (entries[i].action == EACCESS_AUTH_SECURID)
		    {
		      eaccess_unset_auth_securid (r);
		    }
		    else
		    {
		      ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
				   "EAccess: internal error: auth/???");
		      return SERVER_ERROR;
		    }
		    eaccess_log (r, 1, "RE #%03d AUTH removed for '%s'",
		                 i + 1, url);
		  }
		}
		else
		{
		  /*
		   * L'authentification prsente dans le header n'est pas
		   * expire.
		   */
		  eaccess_log (r, 1, "RE #%03d AUTH not expired '%s'",
			       i + 1, url);
		}
	      }
	      else
	      {
	        /*
		 * TTL == 0 => pas d'expiration
		 */
		eaccess_log (r, 1, "RE #%03d AUTH unTTLed for '%s'",
			     i + 1, url);
	      }
	    }
	    else
	    {
	      /*
	       * L'auth prsente dans le Header HTTP n'est pas dj connue.
	       * On mmorise cette premire fois et on continue la boucle
	       * des vrifications d'accs
	       */
	      time (&first_time);
	      eaccess_log (r, 2, "RE #%03d auth/%s: first time this "
			   "Authorization is used (%s/%ld)", i + 1, auth_type,
			   auth, first_time);
	      eaccess_auth_put (r, conf->cachefname, auth, &first_time);
	      eaccess_log (r, 1, "RE #%03d AUTH starting on '%s'",
			   i + 1, url);
	    }
	  }
	  else
	  {
	    eaccess_log (r, 2, "RE #%03d auth/%s: NO Authorization",
	                 i + 1, auth_type);
	    /*
	     * Il n'y a pas d'authentification dans le Header HTTP.
	     * Si l'option dans la rgle avait prcis un realm pour auth/basic
	     * (ou une redirection pour auth/securid), on dit 401 (ou 302 pour
	     * securid), sinon on continue la boucle...
	     */
	    if (entries[i].auth_opt)
	    {
	      if (entries[i].action == EACCESS_AUTH_BASIC)
	      {
		/*
		 * auth/basic:
		 *
		 * Ok, la rgle prcise un realm: on retourne 401 au butineur...
		 */
		eaccess_log (r, 2,
			     "RE #%03d auth/basic: Realm set to '%s' => "
			     "err 401 returned",
			     i + 1, entries[i].auth_opt);
		ap_table_setn (r->err_headers_out, "WWW-Authenticate",
			       ap_pstrcat (r->pool, "Basic realm=\"",
			       ap_escape_quotes (r->pool,
						 entries[i].auth_opt),
			       "\"", NULL));
		eaccess_log (r, 1, "RE #%03d AUTH err 401 for '%s'",
			     i + 1, url);
		r->proxyreq = 0;
		return AUTH_REQUIRED;
	      }
	      else if (entries[i].action == EACCESS_AUTH_SECURID)
	      {
		/*
		 * auth/securid:
		 *
		 * Ok, la rgle prcise une redirection: on retourne 302...
		 */
		eaccess_log (r, 2,
			     "RE #%03d auth/securid: "
			     "Redirect set to '%s' => "
			     "err 302 returned",
			     i + 1, entries[i].auth_opt);
		ap_table_setn (r->headers_out, "Location",
			       ap_pstrdup (r->pool, entries[i].auth_opt));
		eaccess_log (r, 1, "RE #%03d AUTH err 302 for '%s'",
			     i + 1, url);
		/*r->proxyreq = 0;*/
		return REDIRECT;
	      }
	      else
	      {
		/*
		 * ... erreur dans le code ...
		 */
		ap_log_rerror (APLOG_MARK, APLOG_NOERRNO | APLOG_ERR, r,
			     "EAccess: internal error: auth/???");
		return SERVER_ERROR;
	      }
	    }
	    else
	    {
	      /*
	       * La rgle ne prcise pas de realm et pas d'authentification
	       * prsente dans le Header HTTP: on continue...
	       */
	      eaccess_log (r, 1, "RE #%03d AUTH not needed  '%s'",
			   i + 1, url);
	    }
	  }
	}
      }
    }
  }

  /*
   * On est arriv  la dernire ER sans succs, c'est perdu...
   */
  eaccess_log (r, 1, "default denies access to '%s'", url);
  return FORBIDDEN;
}

/*
 ************************************************************************
 * Traitement des commandes de config
 ************************************************************************
 */

/*
 * EAccessEnable
 */
static const char *eaccess_cfg_enable (cmd_parms *cmd, void *dummy, int a1)
{
  /*
   * La config actuelle
   */
  eaccess_cfg *conf =
    (eaccess_cfg *) ap_get_module_config (cmd->server->module_config,
                                          &eaccess_module);
  /*
   * qu'on actualise
   */
  conf->state = (a1 ? EACCESS_ENABLED : EACCESS_DISABLED);

  return NULL;
}

/*
 * EAccessRule
 */
static const char *eaccess_cfg_rule (cmd_parms *cmd, void *dummy, char *a1,
                                     char *a2, char *a3)
{
  /*
   * La config actuelle
   */
  eaccess_cfg *conf =
    (eaccess_cfg *) ap_get_module_config (cmd->server->module_config,
                                          &eaccess_module);
  /*
   * L'action  entreprendre (a1)
   */
  /* eaccess_action	action; */

  /*
   * La RE  compile (a2 ou a2+1 en cas d'utilisation du !)
   */
  char			*regex;

  /*
   * La nouvelle RE compile
   */
  eaccess_rulentry	*new;

  /*
   * On ajoute une entre  la liste de rules
   */
  new = ap_push_array (conf->rules);

  /*
   * Dispatch de l'action
   */
  if      (strcasecmp (a1, EACCESS_PERMIT_STR) == 0)
  {
    new->action = EACCESS_PERMIT;
  }
  else if (strcasecmp (a1, EACCESS_DENY_STR) == 0)
  {
    new->action = EACCESS_DENY;
  }
  else if (strcasecmp (a1, EACCESS_WARNING_STR) == 0)
  {
    new->action = EACCESS_WARNING;
  }
  else if (strncasecmp (a1, EACCESS_AUTH_BASIC_STR,
                        EACCESS_AUTH_BASIC_LEN) == 0)
  {
    new->action   = EACCESS_AUTH_BASIC;
    /*
     * Tant qu'on n'a pas lu l'ventuelle option "realm", on le laisse  null
     */
    new->auth_opt = NULL;
    /*
     * Tant qu'on n'a pas lu l'ventuel TTL, on le laisse  0
     */
    new->auth_ttl = 0;
    /*
     * Si a1 a 1 longueur >  auth/basic, il y a donc 1 '=nnn" derrire
     */
    if (strlen (a1) > EACCESS_AUTH_BASIC_LEN)
    {
      if (a1 [EACCESS_AUTH_BASIC_LEN] == '=')
      {
	new->auth_ttl = atoi (a1 + EACCESS_AUTH_BASIC_LEN + 1);
      }
      else
      {
	return ap_pstrcat (cmd->pool, "EAccess: unknown option in action '",
	                   a1, "'\n", NULL);
      }
    }
    /*
     * a3, facultatif, positionne le "realm" pour auth/basic
     */
    if (a3)
    {
      new->auth_opt = ap_pstrdup (cmd->pool, a3);
    }
  }
  else if (strncasecmp (a1, EACCESS_AUTH_SECURID_STR,
                        EACCESS_AUTH_SECURID_LEN) == 0)
  {
    /*
     * code <=>  EACCESS_AUTH_BASIC
     */
    new->action   = EACCESS_AUTH_SECURID;
    new->auth_opt = NULL;
    new->auth_ttl = 0;
    if (strlen (a1) > EACCESS_AUTH_SECURID_LEN)
    {
      if (a1 [EACCESS_AUTH_SECURID_LEN] == '=')
      {
	new->auth_ttl = atoi (a1 + EACCESS_AUTH_SECURID_LEN + 1);
      }
      else
      {
	return ap_pstrcat (cmd->pool, "EAccess: unknown option in action '",
	                   a1, "'\n", NULL);
      }
    }
    /*
     * a3, facultatif, positionne le "redirect" pour auth/securid
     */
    if (a3)
    {
      new->auth_opt = ap_pstrdup (cmd->pool, a3);
    }

  }
  else
  {
    return ap_pstrcat (cmd->pool, "EAccess: unknown action '", a1, "'\n", NULL);
  }

  /*
   * Si l'ER commence par un !:
   *   - on note qu'on est en revert-match et on saute le !,
   * sinon:
   *   - on note qu'on n'est pas en revert-match
   */
  if (a2 [0] == '!')
  {
    new->revert = TRUE;
    regex = a2 + 1;
  }
  else
  {
    new->revert = FALSE;
    regex = a2;
  }

  /*
   * On compile l'ER avec les options:
   *	- Use POSIX Extended Regular Expression syntax
   *	- Support for substring addressing of matches is not required
   */
  new->regexp = ap_pregcomp (cmd->pool, regex, REG_EXTENDED | REG_NOSUB);
  if (new->regexp == NULL)
  {
    return ap_pstrcat (cmd->pool,
                       "EAccess: cannot compile regular expression '",
		       regex, "'\n", NULL);
  }

  /*
   * On note l'ER en clair (avec le ! ventuel)
   */
  new->pattern = ap_pstrdup (cmd->pool, a2);

  /*
   * Et c'est tout
   */
  return NULL;
}

/*
 * EAccessLog
 */
static const char *eaccess_cfg_log (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * La config actuelle
   */
  eaccess_cfg *conf =
    (eaccess_cfg *) ap_get_module_config (cmd->server->module_config,
                                          &eaccess_module);

  /*
   * On note le nom du fichier
   */
  conf->logfile = a1;

  /*
   * Et c'est tout, on ouvrira le fichier dans le init du module
   */
  return NULL;
}

/*
 * EAccessLogLevel
 */
static const char *eaccess_cfg_loglevel (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * La config actuelle
   */
  eaccess_cfg	*conf =
    (eaccess_cfg *) ap_get_module_config (cmd->server->module_config,
                                          &eaccess_module);

  /*
   * On note le niveau de verbosit
   */
  conf->loglevel = atoi (a1);

  /*
   * Et c'est tout
   */
  return NULL;
}

/*
 * EAccessCache
 */
static const char *eaccess_cfg_cache (cmd_parms *cmd, void *dummy, char *a1)
{
  /*
   * La config actuelle
   */
  eaccess_cfg *conf =
    (eaccess_cfg *) ap_get_module_config (cmd->server->module_config,
                                          &eaccess_module);

  /*
   * On note le nom du fichier
   */
  conf->cachefile = a1;

  /*
   * Et c'est tout, on compltera le nom du fichier dans le init du module
   */
  return NULL;
}

/*
 * Liste
 */
command_rec eaccess_cmds [] =
{
  { "EAccessEnable",	eaccess_cfg_enable,	NULL, RSRC_CONF, TAKE1,
    "On or Off to enable or disable (default) Extended Access control" },
  { "EAccessRule",	eaccess_cfg_rule,	NULL, RSRC_CONF, TAKE23,
    "a action and a [!]URL-applied regexp-pattern (! for revert-match) "
    "and optional options (see docs)" },
  { "EAccessLog",	eaccess_cfg_log,	NULL, RSRC_CONF, TAKE1,
    "the filename of the Extended Access control logfile" },
  { "EAccessLogLevel",	eaccess_cfg_loglevel,	NULL, RSRC_CONF, TAKE1,
    "the level of the Extended Access control logfile verbosity" },
  { "EAccessCache",	eaccess_cfg_cache,	NULL, RSRC_CONF, TAKE1,
    "the filename of the Extended Access control cachefile" },
  {NULL}
};

/*
 ************************************************************************
 * Cration de config
 ************************************************************************
 */
static void *eaccess_create_srv_config (pool *p, server_rec *s)
{
  eaccess_cfg	*conf;

  /*
   * On cre une nouvelle config
   */
  conf = (eaccess_cfg *) ap_pcalloc (p, sizeof (eaccess_cfg));

  /*
   * On initialise les valeurs par dfaut
   */
  conf->state    = EACCESS_DISABLED;
  conf->rules    = ap_make_array (p, 2, sizeof (eaccess_rulentry));
  conf->logfile  = NULL;	/* l'init du module affectera 1 valeur	*/
  conf->logfd    = -1;
  conf->loglevel = 0;

  /*
   * Et on retourne cette config
   */
  return (void *) conf;
}

/*
 ************************************************************************
 * Dfinition du module
 ************************************************************************
 */
module MODULE_VAR_EXPORT eaccess_module =
{
  STANDARD_MODULE_STUFF,
  eaccess_init,			/* initializer */
  NULL,				/* create per-directory config structure */
  NULL,				/* merge per-directory config structures */
  eaccess_create_srv_config,	/* create per-server config structure */
  NULL,				/* merge per-server config structures */
  eaccess_cmds,			/* command table */
  NULL,				/* handlers */
  NULL,				/* translate_handler */
  NULL,				/* check_user_id */
  NULL,				/* check auth */
  eaccess_check,		/* check access */
  NULL,				/* type_checker */
  NULL,				/* pre-run fixups */
  NULL,				/* logger */
  NULL,				/* header parser */
  NULL,				/* child_init */
  NULL,				/* child_exit */
  NULL				/* post read-request */
};
