/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package javax.management;

import java.lang.reflect.Method;

import mx4j.util.Utils;

/**
 * Metadata class for an MBean operation
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.8 $
 */
public class MBeanOperationInfo extends MBeanFeatureInfo
{
   private static final long serialVersionUID = -6178860474881375330L;

   /**
    * This impact means the operation is read-like.
    */
   public static final int INFO = 0;
   /**
    * This impact means the operation is write-like.
    */
   public static final int ACTION = 1;
   /**
    * This impact means the operation is both read-like and write-like.
    */
   public static final int ACTION_INFO = 2;
   /**
    * This impact means the operation impact is unknown.
    */
   public static final int UNKNOWN = 3;

   /**
    * @serial The operation signature
    */
   private MBeanParameterInfo[] signature;
   /**
    * The operation return type
    */
   private String type;
   /**
    * The operation impact
    */
   private int impact;

   /**
    * Creates a new MBeanOperationInfo.
    * @param description The operation description
    * @param method The method
    */
   public MBeanOperationInfo(String description, Method method)
   {
      super(method.getName(), description);
      Class[] params = method.getParameterTypes();
      this.signature = new MBeanParameterInfo[params.length];
      for (int i = 0; i < params.length; ++i)
      {
         this.signature[i] = new MBeanParameterInfo("", params[i].getName(), "");
      }
      this.type = method.getReturnType().getName();
      this.impact = UNKNOWN;
   }

   /**
    * Creates a new MBeanOperationInfo
    * @param name The operation name
    * @param description The operation description
    * @param signature The operation signature
    * @param type The operation return type
    * @param impact The operation impact
    */
   public MBeanOperationInfo(String name, String description, MBeanParameterInfo[] signature, String type, int impact)
   {
      super(name, description);
      this.signature = signature == null ? new MBeanParameterInfo[0] : signature;
      this.type = type;
      this.impact = impact;
   }

   /**
    * Returns the return type of the operation
    */
   public String getReturnType()
   {
      return type;
   }

   /**
    * Returns the signature of the operation
    */
   public MBeanParameterInfo[] getSignature()
   {
      return signature;
   }

   /**
    * Returns the impact of the operation
    */
   public int getImpact()
   {
      return impact;
   }

   public int hashCode()
   {
      int hash = super.hashCode();

      String type = getReturnType();
      hash ^= type == null ? 0 : type.hashCode();

      hash ^= Utils.arrayHashCode(getSignature());

      return hash;
   }

   public boolean equals(Object obj)
   {
      if (!super.equals(obj)) return false;

      try
      {
         MBeanOperationInfo other = (MBeanOperationInfo)obj;

         String type = getReturnType();
         String otherType = other.getReturnType();
         if (type != null && !type.equals(otherType)) return false;
         if (type == null && otherType != null) return false;

         // No checks on description and impact

         return Utils.arrayEquals(getSignature(), other.getSignature());
      }
      catch (ClassCastException ignored)
      {
      }
      return false;
   }
}
